// API服务配置
let API_BASE_URL = 'http://localhost:8080/api';

// 如果是生产环境，使用域名
if (process.env.NODE_ENV === 'production') {
  API_BASE_URL = 'http://likeusewin10.xyz/api';
}

// 通用请求函数
async function request(url, options = {}) {
  const config = {
    headers: {
      ...options.headers
    },
    ...options
  };

  // 如果不是文件上传，设置JSON Content-Type
  if (!options.body || !(options.body instanceof FormData)) {
    config.headers['Content-Type'] = 'application/json';
  }

  // 添加认证token
  const token = localStorage.getItem('token');
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }

  try {
    const response = await fetch(`${API_BASE_URL}${url}`, config);
    
    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    
    const data = await response.json();
    return data;
  } catch (error) {
    console.error('API请求失败:', error);
    throw error;
  }
}

// 用户相关API
export const userAPI = {
  // 用户登录
  login: (loginData) => {
    return request('/user/login', {
      method: 'POST',
      body: JSON.stringify(loginData)
    });
  },

  // 用户注册
  register: (userData) => {
    return request('/user/register', {
      method: 'POST',
      body: JSON.stringify(userData)
    });
  },

  // 发送忘记密码验证码
  forgotPassword: (email) => {
    return request('/user/forgot-password', {
      method: 'POST',
      body: JSON.stringify({ email })
    });
  },

  // 验证邮箱验证码
  verifyCode: (email, verificationCode) => {
    return request('/user/verify-code', {
      method: 'POST',
      body: JSON.stringify({ email, verificationCode })
    });
  },

  // 重置密码
  resetPassword: (email, verificationCode, newPassword) => {
    return request('/user/reset-password', {
      method: 'POST',
      body: JSON.stringify({ email, verificationCode, newPassword })
    });
  },

  // 获取用户信息
  getUserInfo: () => {
    return request('/user/info');
  },

  // 更新用户信息
  updateUserInfo: (userData) => {
    return request('/user/info', {
      method: 'PUT',
      body: JSON.stringify(userData)
    });
  },

  // 修改密码
  updatePassword: (passwordData) => {
    return request('/user/password', {
      method: 'PUT',
      body: JSON.stringify(passwordData)
    });
  }
};

// 文件上传相关API
export const uploadAPI = {
  // 上传头像
  uploadAvatar: (file) => {
    const formData = new FormData();
    formData.append('file', file);
    
    return request('/upload/avatar', {
      method: 'POST',
      body: formData
    });
  },
  
  // 上传文章图片
  uploadArticleImage: (file) => {
    const formData = new FormData();
    formData.append('file', file);
    
    return request('/upload/article-image', {
      method: 'POST',
      body: formData
    });
  },
  
  // 上传评论图片
  uploadCommentImage: (file) => {
    const formData = new FormData();
    formData.append('file', file);
    
    return request('/upload/comment-image', {
      method: 'POST',
      body: formData
    });
  }
};

// 游戏相关API
export const gameAPI = {
  // 获取所有游戏
  getAllGames: () => {
    return request('/game/videos');
  },

  // 根据ID获取游戏
  getGameById: (id) => {
    return request(`/game/videos/${id}`);
  },

  // 搜索游戏
  searchGames: (keyword) => {
    return request(`/game/search?keyword=${encodeURIComponent(keyword)}`);
  }
};

// 音乐相关API
export const musicAPI = {
  // 获取所有音乐
  getAllMusic: () => {
    return request('/music/list');
  },

  // 根据ID获取音乐
  getMusicById: (id) => {
    return request(`/music/${id}`);
  },

  // 搜索音乐
  searchMusic: (keyword) => {
    return request(`/music/search?keyword=${encodeURIComponent(keyword)}`);
  }
};

// 文章相关API
export const articleAPI = {
  // 获取所有文章
  getAllArticles: (page = 1, size = 10) => {
    return request(`/article/list?current=${page}&size=${size}`);
  },

  // 根据ID获取文章
  getArticleById: (id) => {
    return request(`/article/${id}`);
  },

  // 创建文章
  createArticle: (articleData) => {
    return request('/article', {
      method: 'POST',
      body: JSON.stringify(articleData)
    });
  },

  // 更新文章
  updateArticle: (id, articleData) => {
    return request(`/article/${id}`, {
      method: 'PUT',
      body: JSON.stringify(articleData)
    });
  },

  // 删除文章
  deleteArticle: (id) => {
    return request(`/article/${id}`, {
      method: 'DELETE'
    });
  },

  // 搜索文章
  searchArticles: (keyword, page = 1, size = 10) => {
    return request(`/article/search?keyword=${encodeURIComponent(keyword)}&current=${page}&size=${size}`);
  },

  // 获取用户文章
  getUserArticles: (page = 1, size = 10) => {
    return request(`/article/my-articles?current=${page}&size=${size}`);
  }
};

// 评论相关API
export const commentAPI = {
  // 获取文章评论
  getArticleComments: (articleId, page = 1, size = 10) => {
    return request(`/comment/article/${articleId}?current=${page}&size=${size}`);
  },

  // 创建评论
  createComment: (commentData) => {
    return request('/comment', {
      method: 'POST',
      body: JSON.stringify(commentData)
    });
  },

  // 删除评论
  deleteComment: (id) => {
    return request(`/comment/${id}`, {
      method: 'DELETE'
    });
  }
};

// 通用搜索API
export const searchAPI = {
  // 全局搜索
  globalSearch: (keyword, type = 'all', page = 1, size = 10) => {
    return request(`/search?keyword=${encodeURIComponent(keyword)}&type=${type}&page=${page}&size=${size}`);
  }
};

export default {
  userAPI,
  uploadAPI,
  gameAPI,
  musicAPI,
  articleAPI,
  commentAPI,
  searchAPI
};