// API测试脚本
const API_BASE_URL = 'http://localhost:8080/api';

// 通用请求函数
async function request(url, options = {}) {
  const config = {
    headers: {
      'Content-Type': 'application/json',
      ...options.headers
    },
    ...options
  };

  try {
    const response = await fetch(`${API_BASE_URL}${url}`, config);
    const data = await response.json();
    return {
      success: response.ok,
      status: response.status,
      data: data
    };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}

// 测试所有API接口
async function testAllAPIs() {
  console.log('开始测试所有API接口...');
  console.log('=' .repeat(50));

  // 1. 测试健康检查接口
  console.log('\n1. 测试健康检查接口');
  const healthResult = await request('/test/health');
  console.log('结果:', healthResult);

  // 2. 测试游戏相关API
  console.log('\n2. 测试游戏相关API');
  console.log('2.1 获取游戏视频列表');
  const gameVideosResult = await request('/game/videos');
  console.log('结果:', gameVideosResult);

  console.log('2.2 获取游戏分类');
  const gameCategoriesResult = await request('/game/categories');
  console.log('结果:', gameCategoriesResult);

  // 3. 测试音乐相关API
  console.log('\n3. 测试音乐相关API');
  console.log('3.1 获取音乐列表');
  const musicListResult = await request('/music/list');
  console.log('结果:', musicListResult);

  console.log('3.2 获取热门音乐');
  const hotMusicResult = await request('/music/hot');
  console.log('结果:', hotMusicResult);

  // 4. 测试文章相关API
  console.log('\n4. 测试文章相关API');
  console.log('4.1 获取文章列表');
  const articleListResult = await request('/article/list');
  console.log('结果:', articleListResult);

  // 5. 测试用户相关API（不需要认证的）
  console.log('\n5. 测试用户相关API');
  console.log('5.1 测试登录接口（错误凭据）');
  const loginResult = await request('/user/login', {
    method: 'POST',
    body: JSON.stringify({
      username: 'test',
      password: 'test'
    })
  });
  console.log('结果:', loginResult);

  // 6. 生成测试报告
  console.log('\n' + '=' .repeat(50));
  console.log('API测试报告:');
  console.log('=' .repeat(50));
  
  const testResults = [
    { name: '健康检查', result: healthResult },
    { name: '游戏视频列表', result: gameVideosResult },
    { name: '游戏分类', result: gameCategoriesResult },
    { name: '音乐列表', result: musicListResult },
    { name: '热门音乐', result: hotMusicResult },
    { name: '文章列表', result: articleListResult },
    { name: '用户登录', result: loginResult }
  ];

  testResults.forEach(test => {
    const status = test.result.success ? '✅ 正常' : '❌ 异常';
    const statusCode = test.result.status || 'N/A';
    console.log(`${test.name}: ${status} (状态码: ${statusCode})`);
    
    if (!test.result.success) {
      console.log(`  错误信息: ${test.result.error || test.result.data?.message || '未知错误'}`);
    }
  });

  // 统计
  const successCount = testResults.filter(t => t.result.success).length;
  const totalCount = testResults.length;
  console.log(`\n测试完成: ${successCount}/${totalCount} 个接口正常`);
}

// 如果是在Node.js环境中运行
if (typeof window === 'undefined') {
  // Node.js环境，需要安装node-fetch
  try {
    // 使用动态导入来支持ES模块
    import('node-fetch').then(({ default: fetch }) => {
      global.fetch = fetch;
      testAllAPIs();
    }).catch(error => {
      console.log('请先安装node-fetch: npm install node-fetch');
      console.log('或者在浏览器控制台中运行此脚本');
    });
  } catch (error) {
    console.log('请先安装node-fetch: npm install node-fetch');
    console.log('或者在浏览器控制台中运行此脚本');
  }
} else {
  // 浏览器环境
  testAllAPIs();
}

// 导出测试函数供浏览器使用
if (typeof window !== 'undefined') {
  window.testAllAPIs = testAllAPIs;
}