package com.likeusewin10.controller;

import com.likeusewin10.dto.request.ArticleRequest;
import com.likeusewin10.dto.response.ArticleResponse;
import com.likeusewin10.dto.response.PageResult;
import com.likeusewin10.dto.response.Result;
import com.likeusewin10.service.ArticleService;
import com.likeusewin10.util.JwtUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;

@RestController
@RequestMapping("/api/article")
public class ArticleController {
    
    @Autowired
    private ArticleService articleService;
    
    /**
     * 获取文章列表
     */
    @GetMapping("/list")
    public Result<PageResult<ArticleResponse>> getArticleList(
            @RequestParam(defaultValue = "1") int current,
            @RequestParam(defaultValue = "10") int size) {
        return articleService.getArticleList(current, size);
    }
    
    /**
     * 获取用户文章列表
     */
    @GetMapping("/my-articles")
    public Result<PageResult<ArticleResponse>> getUserArticles(
            HttpServletRequest request,
            @RequestParam(defaultValue = "1") int current,
            @RequestParam(defaultValue = "10") int size) {
        try {
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7); // 移除 "Bearer " 前缀
            Long userId = JwtUtil.getUserIdFromToken(token);
            
            return articleService.getUserArticles(userId, current, size);
        } catch (Exception e) {
            return Result.error("获取用户文章失败：" + e.getMessage());
        }
    }
    
    /**
     * 获取文章详情
     */
    @GetMapping("/{id}")
    public Result<ArticleResponse> getArticleById(@PathVariable Long id) {
        return articleService.getArticleById(id);
    }
    
    /**
     * 创建文章
     */
    @PostMapping
    public Result<Void> createArticle(HttpServletRequest request, @RequestBody ArticleRequest articleRequest) {
        try {
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7); // 移除 "Bearer " 前缀
            Long userId = JwtUtil.getUserIdFromToken(token);
            articleRequest.setAuthorId(userId);
            
            return articleService.createArticle(userId, articleRequest);
        } catch (Exception e) {
            return Result.error("创建文章失败：" + e.getMessage());
        }
    }
    
    /**
     * 更新文章
     */
    @PutMapping("/{id}")
    public Result<Void> updateArticle(HttpServletRequest request, @PathVariable Long id, @RequestBody ArticleRequest articleRequest) {
        try {
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7); // 移除 "Bearer " 前缀
            Long userId = JwtUtil.getUserIdFromToken(token);
            
            // 这里可以添加权限检查，确保只有文章作者可以更新
            
            return articleService.updateArticle(userId, id, articleRequest);
        } catch (Exception e) {
            return Result.error("更新文章失败：" + e.getMessage());
        }
    }
    
    /**
     * 删除文章
     */
    @DeleteMapping("/{id}")
    public Result<Void> deleteArticle(HttpServletRequest request, @PathVariable Long id) {
        try {
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7); // 移除 "Bearer " 前缀
            Long userId = JwtUtil.getUserIdFromToken(token);
            
            // 这里可以添加权限检查，确保只有文章作者可以删除
            
            return articleService.deleteArticle(userId, id);
        } catch (Exception e) {
            return Result.error("删除文章失败：" + e.getMessage());
        }
    }
    
    /**
     * 搜索文章
     */
    @GetMapping("/search")
    public Result<PageResult<ArticleResponse>> searchArticles(
            @RequestParam String keyword,
            @RequestParam(defaultValue = "1") int current,
            @RequestParam(defaultValue = "10") int size) {
        return articleService.searchArticles(keyword, current, size);
    }
}