package com.likeusewin10.controller;

import com.likeusewin10.dto.request.CommentRequest;
import com.likeusewin10.dto.response.CommentResponse;
import com.likeusewin10.dto.response.PageResult;
import com.likeusewin10.dto.response.Result;
import com.likeusewin10.service.CommentService;
import com.likeusewin10.util.JwtUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;

@RestController
@RequestMapping("/api/comment")
public class CommentController {
    
    @Autowired
    private CommentService commentService;
    
    /**
     * 获取文章评论列表
     */
    @GetMapping("/article/{articleId}")
    public Result<PageResult<CommentResponse>> getCommentsByArticleId(
            @PathVariable Long articleId,
            @RequestParam(defaultValue = "1") int current,
            @RequestParam(defaultValue = "10") int size) {
        return commentService.getCommentsByArticleId(articleId, current, size);
    }
    
    /**
     * 创建评论
     */
    @PostMapping
    public Result<Void> createComment(HttpServletRequest request, @RequestBody CommentRequest commentRequest) {
        try {
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7); // 移除 "Bearer " 前缀
            Long userId = JwtUtil.getUserIdFromToken(token);
            commentRequest.setUserId(userId);
            
            return commentService.createComment(userId, commentRequest);
        } catch (Exception e) {
            return Result.error("创建评论失败：" + e.getMessage());
        }
    }
    
    /**
     * 删除评论
     */
    @DeleteMapping("/{id}")
    public Result<Void> deleteComment(HttpServletRequest request, @PathVariable Long id) {
        try {
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7); // 移除 "Bearer " 前缀
            Long userId = JwtUtil.getUserIdFromToken(token);
            
            return commentService.deleteComment(id, userId);
        } catch (Exception e) {
            return Result.error("删除评论失败：" + e.getMessage());
        }
    }
}