package com.likeusewin10.controller;

import com.likeusewin10.dto.response.Result;
import com.likeusewin10.entity.User;
import com.likeusewin10.mapper.UserMapper;
import com.likeusewin10.util.JwtUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * 文件上传控制器
 * 
 * @author likeusewin10
 * @since 2025-01-10
 */
@RestController
@RequestMapping("/api/upload")
public class FileUploadController {
    
    @Autowired
    private UserMapper userMapper;
    
    // 上传文件根目录 - 保存到后端uploads目录
    private static final String UPLOAD_ROOT = System.getProperty("user.dir") + File.separator + "uploads";
    
    // 允许的图片格式
    private static final String[] ALLOWED_IMAGE_TYPES = {
        "image/jpeg", "image/jpg", "image/png", "image/gif", "image/webp"
    };
    
    // 文件大小限制（5MB）
    private static final long MAX_FILE_SIZE = 5 * 1024 * 1024;
    
    /**
     * 上传用户头像
     */
    @PostMapping("/avatar")
    public Result<Map<String, Object>> uploadAvatar(
            @RequestParam("file") MultipartFile file,
            HttpServletRequest request) {
        
        try {
            // 验证用户身份
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7);
            Long userId = JwtUtil.getUserIdFromToken(token);
            
            // 验证文件
            String validationResult = validateImageFile(file);
            if (validationResult != null) {
                return Result.error(validationResult);
            }
            
            // 生成文件路径
            String fileName = generateFileName(file.getOriginalFilename(), "avatar", userId);
            String relativePath = "avatars/" + fileName;
            String fullPath = UPLOAD_ROOT + "/" + relativePath;
            
            // 保存文件
            saveFile(file, fullPath);
            
            // 更新用户头像URL到数据库
            String avatarUrl = "/uploads/" + relativePath;
            User user = userMapper.selectById(userId);
            if (user != null) {
                user.setAvatarUrl(avatarUrl);
                user.setUpdatedAt(LocalDateTime.now());
                userMapper.updateById(user);
            }
            
            // 返回文件访问URL
            Map<String, Object> response = new HashMap<>();
            response.put("url", avatarUrl);
            response.put("fileName", fileName);
            response.put("fileSize", file.getSize());
            
            return Result.success("头像上传成功", response);
            
        } catch (Exception e) {
            return Result.error("头像上传失败：" + e.getMessage());
        }
    }
    
    /**
     * 上传论坛文章图片
     */
    @PostMapping("/article-image")
    public Result<Map<String, Object>> uploadArticleImage(
            @RequestParam("file") MultipartFile file,
            HttpServletRequest request) {
        
        try {
            // 验证用户身份
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7);
            Long userId = JwtUtil.getUserIdFromToken(token);
            
            // 验证文件
            String validationResult = validateImageFile(file);
            if (validationResult != null) {
                return Result.error(validationResult);
            }
            
            // 生成文件路径
            String fileName = generateFileName(file.getOriginalFilename(), "article", userId);
            String relativePath = "articles/" + fileName;
            String fullPath = UPLOAD_ROOT + "/" + relativePath;
            
            // 保存文件
            saveFile(file, fullPath);
            
            // 返回文件访问URL
            Map<String, Object> response = new HashMap<>();
            response.put("url", "/uploads/" + relativePath);
            response.put("fileName", fileName);
            response.put("fileSize", file.getSize());
            
            return Result.success("文章图片上传成功", response);
            
        } catch (Exception e) {
            return Result.error("文章图片上传失败：" + e.getMessage());
        }
    }
    
    /**
     * 上传评论图片
     */
    @PostMapping("/comment-image")
    public Result<Map<String, Object>> uploadCommentImage(
            @RequestParam("file") MultipartFile file,
            HttpServletRequest request) {
        
        try {
            // 验证用户身份
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7);
            Long userId = JwtUtil.getUserIdFromToken(token);
            
            // 验证文件
            String validationResult = validateImageFile(file);
            if (validationResult != null) {
                return Result.error(validationResult);
            }
            
            // 生成文件路径
            String fileName = generateFileName(file.getOriginalFilename(), "comment", userId);
            String relativePath = "comments/" + fileName;
            String fullPath = UPLOAD_ROOT + "/" + relativePath;
            
            // 保存文件
            saveFile(file, fullPath);
            
            // 返回文件访问URL
            Map<String, Object> response = new HashMap<>();
            response.put("url", "/uploads/" + relativePath);
            response.put("fileName", fileName);
            response.put("fileSize", file.getSize());
            
            return Result.success("评论图片上传成功", response);
            
        } catch (Exception e) {
            return Result.error("评论图片上传失败：" + e.getMessage());
        }
    }
    
    /**
     * 验证图片文件
     */
    private String validateImageFile(MultipartFile file) {
        if (file == null || file.isEmpty()) {
            return "请选择要上传的文件";
        }
        
        // 检查文件大小
        if (file.getSize() > MAX_FILE_SIZE) {
            return "文件大小不能超过5MB";
        }
        
        // 检查文件类型
        String contentType = file.getContentType();
        boolean isValidType = false;
        for (String allowedType : ALLOWED_IMAGE_TYPES) {
            if (allowedType.equals(contentType)) {
                isValidType = true;
                break;
            }
        }
        
        if (!isValidType) {
            return "只支持JPG、PNG、GIF、WEBP格式的图片";
        }
        
        return null; // 验证通过
    }
    
    /**
     * 生成文件名
     */
    private String generateFileName(String originalFilename, String type, Long userId) {
        // 获取文件扩展名
        String extension = "";
        if (originalFilename != null && originalFilename.contains(".")) {
            extension = originalFilename.substring(originalFilename.lastIndexOf("."));
        }
        
        // 对于头像文件，使用简单的用户ID命名，便于前端获取
        if ("avatar".equals(type)) {
            return String.format("user_%d%s", userId, extension);
        }
        
        // 对于其他文件，保持原有的复杂命名规则
        // 生成时间戳
        String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss"));
        
        // 生成UUID（取前8位）
        String uuid = UUID.randomUUID().toString().substring(0, 8);
        
        // 组合文件名：类型_用户ID_时间戳_UUID.扩展名
        return String.format("%s_%d_%s_%s%s", type, userId, timestamp, uuid, extension);
    }
    
    /**
     * 保存文件
     */
    private void saveFile(MultipartFile file, String fullPath) throws IOException {
        Path path = Paths.get(fullPath);
        
        // 创建目录（如果不存在）
        Files.createDirectories(path.getParent());
        
        // 保存文件
        file.transferTo(path.toFile());
    }
}