package com.likeusewin10.controller;

import com.likeusewin10.dto.request.LoginRequest;
import com.likeusewin10.dto.request.RegisterRequest;
import com.likeusewin10.dto.request.ForgotPasswordRequest;
import com.likeusewin10.dto.request.VerifyCodeRequest;
import com.likeusewin10.dto.request.ResetPasswordRequest;
import com.likeusewin10.dto.request.UpdatePasswordRequest;
import com.likeusewin10.dto.response.Result;
import com.likeusewin10.dto.response.UserResponse;
import com.likeusewin10.dto.response.LoginResponse;
import com.likeusewin10.service.UserService;
import com.likeusewin10.service.CaptchaService;
import com.likeusewin10.util.JwtUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;

@RestController
@RequestMapping("/api/user")
public class UserController {
    
    @Autowired
    private UserService userService;
    
    @Autowired
    private CaptchaService captchaService;
    
    /**
     * 用户登录
     */
    @PostMapping("/login")
    public Result<LoginResponse> login(@RequestBody LoginRequest request, HttpServletRequest httpRequest) {
        return userService.login(request, httpRequest);
    }
    
    /**
     * 用户注册
     */
    @PostMapping("/register")
    public Result<Void> register(@RequestBody RegisterRequest request, HttpServletRequest httpRequest) {
        return userService.register(request, httpRequest);
    }
    
    /**
     * 发送忘记密码验证码
     */
    @PostMapping("/forgot-password")
    public Result<Void> forgotPassword(@RequestBody ForgotPasswordRequest request) {
        return userService.sendForgotPasswordCode(request);
    }
    
    /**
     * 验证邮箱验证码
     */
    @PostMapping("/verify-code")
    public Result<Void> verifyCode(@RequestBody VerifyCodeRequest request) {
        return userService.verifyEmailCode(request);
    }
    
    /**
     * 重置密码
     */
    @PostMapping("/reset-password")
    public Result<Void> resetPassword(@RequestBody ResetPasswordRequest request) {
        return userService.resetPassword(request);
    }
    
    /**
     * 获取用户信息
     */
    @GetMapping("/info")
    public Result<UserResponse> getUserInfo(HttpServletRequest request) {
        try {
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7); // 移除 "Bearer " 前缀
            Long userId = JwtUtil.getUserIdFromToken(token);
            
            return userService.getUserInfo(userId);
        } catch (Exception e) {
            return Result.error("获取用户信息失败：" + e.getMessage());
        }
    }
    
    /**
     * 检查用户名是否存在
     */
    @GetMapping("/check-username/{username}")
    public Result<Boolean> checkUsername(@PathVariable String username) {
        return userService.checkUsernameExists(username);
    }
    
    /**
     * 检查邮箱是否存在
     */
    @GetMapping("/check-email/{email}")
    public Result<Boolean> checkEmail(@PathVariable String email) {
        return userService.checkEmailExists(email);
    }
    
    /**
     * 更新用户信息
     */
    @PutMapping("/info")
    public Result<Void> updateUserInfo(HttpServletRequest request, @RequestBody Map<String, Object> requestData) {
        try {
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7); // 移除 "Bearer " 前缀
            Long userId = JwtUtil.getUserIdFromToken(token);
            
            // 验证验证码
            String captchaId = (String) requestData.get("captchaId");
            String captcha = (String) requestData.get("captcha");
            
            if (captchaId == null || captcha == null) {
                return Result.error("请输入验证码");
            }
            
            if (!captchaService.verifyCaptcha(captchaId, captcha, request)) {
                return Result.error("验证码错误或已过期");
            }
            
            // 提取用户信息
            UserResponse userInfo = new UserResponse();
            userInfo.setNickname((String) requestData.get("nickname"));
            userInfo.setEmail((String) requestData.get("email"));
            userInfo.setAvatarUrl((String) requestData.get("avatarUrl"));
            userInfo.setGender((String) requestData.get("gender"));
            userInfo.setBirthday((String) requestData.get("birthday"));
            userInfo.setLocation((String) requestData.get("location"));
            userInfo.setBio((String) requestData.get("bio"));
            
            return userService.updateUserInfo(userId, userInfo);
        } catch (Exception e) {
            return Result.error("更新用户信息失败：" + e.getMessage());
        }
    }
    
    /**
     * 修改密码
     */
    @PutMapping("/password")
    public Result<Void> updatePassword(HttpServletRequest request, @RequestBody UpdatePasswordRequest updatePasswordRequest) {
        try {
            String token = request.getHeader("Authorization");
            if (token == null || !token.startsWith("Bearer ")) {
                return Result.error("未提供有效的认证令牌");
            }
            
            token = token.substring(7); // 移除 "Bearer " 前缀
            Long userId = JwtUtil.getUserIdFromToken(token);
            
            return userService.updatePassword(userId, updatePasswordRequest, request);
        } catch (Exception e) {
            return Result.error("修改密码失败：" + e.getMessage());
        }
    }
}