package com.likeusewin10.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.likeusewin10.dto.request.ArticleRequest;
import com.likeusewin10.dto.response.ArticleResponse;
import com.likeusewin10.dto.response.PageResult;
import com.likeusewin10.dto.response.Result;
import com.likeusewin10.entity.Article;
import com.likeusewin10.entity.User;
import com.likeusewin10.mapper.ArticleMapper;
import com.likeusewin10.mapper.UserMapper;
import com.likeusewin10.service.ArticleService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

@Service
public class ArticleServiceImpl implements ArticleService {
    
    @Autowired
    private ArticleMapper articleMapper;
    
    @Autowired
    private UserMapper userMapper;
    
    @Override
    public Result<PageResult<ArticleResponse>> getArticleList(Integer current, Integer size) {
        try {
            Page<Article> page = new Page<>(current, size);
            QueryWrapper<Article> queryWrapper = new QueryWrapper<>();
            
            queryWrapper.eq("status", 1); // 只查询已发布的文章
            queryWrapper.orderByDesc("created_at");
            
            Page<Article> articlePage = articleMapper.selectPage(page, queryWrapper);
            
            List<ArticleResponse> articleResponses = articlePage.getRecords().stream()
                    .map(this::convertToResponse)
                    .collect(Collectors.toList());
            
            PageResult<ArticleResponse> pageResult = new PageResult<>();
            pageResult.setRecords(articleResponses);
            pageResult.setTotal(articlePage.getTotal());
            pageResult.setCurrent(articlePage.getCurrent());
            pageResult.setSize(articlePage.getSize());
            pageResult.setPages(articlePage.getPages());
            
            return Result.success(pageResult);
        } catch (Exception e) {
            return Result.error("获取文章列表失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<ArticleResponse> getArticleById(Long id) {
        try {
            Article article = articleMapper.selectById(id);
            if (article == null) {
                return Result.error("文章不存在");
            }
            
            // 移除浏览量功能
            
            ArticleResponse response = convertToResponse(article);
            return Result.success(response);
        } catch (Exception e) {
            return Result.error("获取文章详情失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> createArticle(Long userId, ArticleRequest request) {
        try {
            Article article = new Article();
            BeanUtils.copyProperties(request, article);
            article.setAuthorId(userId);
            article.setCreatedAt(LocalDateTime.now());
            article.setUpdatedAt(LocalDateTime.now());
            // 移除浏览量字段设置
            article.setLikeCount(0);
            article.setStatus(1);
            
            articleMapper.insert(article);
            return Result.success();
        } catch (Exception e) {
            return Result.error("创建文章失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> updateArticle(Long userId, Long articleId, ArticleRequest request) {
        try {
            Article article = articleMapper.selectById(articleId);
            if (article == null) {
                return Result.error("文章不存在");
            }
            
            // 验证文章作者
            if (!article.getAuthorId().equals(userId)) {
                return Result.error("只能修改自己的文章");
            }
            
            BeanUtils.copyProperties(request, article);
            article.setUpdatedAt(LocalDateTime.now());
            
            articleMapper.updateById(article);
            return Result.success();
        } catch (Exception e) {
            return Result.error("更新文章失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> deleteArticle(Long userId, Long articleId) {
        try {
            Article article = articleMapper.selectById(articleId);
            if (article == null) {
                return Result.error("文章不存在");
            }
            
            // 验证文章作者
            if (!article.getAuthorId().equals(userId)) {
                return Result.error("只能删除自己的文章");
            }
            
            articleMapper.deleteById(articleId);
            return Result.success();
        } catch (Exception e) {
            return Result.error("删除文章失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<PageResult<ArticleResponse>> searchArticles(String keyword, Integer current, Integer size) {
        try {
            Page<Article> page = new Page<>(current, size);
            QueryWrapper<Article> queryWrapper = new QueryWrapper<>();
            
            if (keyword != null && !keyword.trim().isEmpty()) {
                queryWrapper.like("title", keyword)
                           .or()
                           .like("content", keyword)
                           .or()
                           .like("summary", keyword);
            }
            
            queryWrapper.eq("status", 1); // 只查询已发布的文章
            queryWrapper.orderByDesc("created_at");
            
            Page<Article> articlePage = articleMapper.selectPage(page, queryWrapper);
            
            List<ArticleResponse> articleResponses = articlePage.getRecords().stream()
                    .map(this::convertToResponse)
                    .collect(Collectors.toList());
            
            PageResult<ArticleResponse> pageResult = new PageResult<>();
            pageResult.setRecords(articleResponses);
            pageResult.setTotal(articlePage.getTotal());
            pageResult.setCurrent(articlePage.getCurrent());
            pageResult.setSize(articlePage.getSize());
            pageResult.setPages(articlePage.getPages());
            
            return Result.success(pageResult);
        } catch (Exception e) {
            return Result.error("搜索文章失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<PageResult<ArticleResponse>> getUserArticles(Long userId, Integer current, Integer size) {
        try {
            Page<Article> page = new Page<>(current, size);
            QueryWrapper<Article> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("author_id", userId)
                       .orderByDesc("created_at");
            
            Page<Article> articlePage = articleMapper.selectPage(page, queryWrapper);
            
            List<ArticleResponse> articleResponses = articlePage.getRecords().stream()
                    .map(this::convertToResponse)
                    .collect(Collectors.toList());
            
            PageResult<ArticleResponse> pageResult = new PageResult<>();
            pageResult.setRecords(articleResponses);
            pageResult.setTotal(articlePage.getTotal());
            pageResult.setCurrent(articlePage.getCurrent());
            pageResult.setSize(articlePage.getSize());
            pageResult.setPages(articlePage.getPages());
            
            return Result.success(pageResult);
        } catch (Exception e) {
            return Result.error("获取用户文章失败：" + e.getMessage());
        }
    }
    
    private ArticleResponse convertToResponse(Article article) {
        ArticleResponse response = new ArticleResponse();
        BeanUtils.copyProperties(article, response);
        
        // 获取作者信息
        User author = userMapper.selectById(article.getAuthorId());
        if (author != null) {
            response.setAuthorName(author.getNickname() != null ? author.getNickname() : author.getUsername());
        }
        
        return response;
    }
}