package com.likeusewin10.service.impl;

import com.likeusewin10.service.CaptchaService;
import org.springframework.stereotype.Service;

import javax.imageio.ImageIO;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

@Service
public class CaptchaServiceImpl implements CaptchaService {
    
    // 验证码存储（生产环境建议使用Redis）
    private static final Map<String, CaptchaInfo> captchaStore = new ConcurrentHashMap<>();
    
    // 验证码字符集
    private static final String CAPTCHA_CHARS = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    
    // 验证码长度
    private static final int CAPTCHA_LENGTH = 4;
    
    // 验证码有效期（5分钟）
    private static final long CAPTCHA_EXPIRE_TIME = 5 * 60 * 1000;
    
    @Override
    public Map<String, Object> generateCaptcha(HttpServletRequest request) {
        // 生成验证码文本
        String captchaText = generateCaptchaText();
        
        // 生成验证码ID
        String captchaId = UUID.randomUUID().toString();
        
        // 生成验证码图片
        String imageBase64 = generateCaptchaImage(captchaText);
        
        // 存储验证码信息
        CaptchaInfo captchaInfo = new CaptchaInfo(captchaText, System.currentTimeMillis());
        captchaStore.put(captchaId, captchaInfo);
        
        // 清理过期验证码
        cleanExpiredCaptcha();
        
        Map<String, Object> result = new HashMap<>();
        result.put("id", captchaId);
        result.put("image", "data:image/png;base64," + imageBase64);
        
        return result;
    }
    
    @Override
    public boolean verifyCaptcha(String captchaId, String captcha, HttpServletRequest request) {
        if (captchaId == null || captcha == null) {
            return false;
        }
        
        CaptchaInfo captchaInfo = captchaStore.get(captchaId);
        if (captchaInfo == null) {
            return false;
        }
        
        // 检查是否过期
        if (System.currentTimeMillis() - captchaInfo.getCreateTime() > CAPTCHA_EXPIRE_TIME) {
            captchaStore.remove(captchaId);
            return false;
        }
        
        // 验证码只能使用一次
        captchaStore.remove(captchaId);
        
        // 忽略大小写比较
        return captchaInfo.getText().equalsIgnoreCase(captcha);
    }
    
    /**
     * 生成验证码文本
     */
    private String generateCaptchaText() {
        Random random = new Random();
        StringBuilder sb = new StringBuilder();
        
        for (int i = 0; i < CAPTCHA_LENGTH; i++) {
            int index = random.nextInt(CAPTCHA_CHARS.length());
            sb.append(CAPTCHA_CHARS.charAt(index));
        }
        
        return sb.toString();
    }
    
    /**
     * 生成验证码图片
     */
    private String generateCaptchaImage(String captchaText) {
        int width = 100;
        int height = 40;
        
        BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
        Graphics2D g2d = image.createGraphics();
        
        // 设置抗锯齿
        g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        
        // 填充背景
        g2d.setColor(Color.WHITE);
        g2d.fillRect(0, 0, width, height);
        
        // 绘制干扰线
        Random random = new Random();
        g2d.setColor(Color.LIGHT_GRAY);
        for (int i = 0; i < 5; i++) {
            int x1 = random.nextInt(width);
            int y1 = random.nextInt(height);
            int x2 = random.nextInt(width);
            int y2 = random.nextInt(height);
            g2d.drawLine(x1, y1, x2, y2);
        }
        
        // 绘制验证码文字
        g2d.setFont(new Font("Arial", Font.BOLD, 20));
        for (int i = 0; i < captchaText.length(); i++) {
            // 随机颜色
            g2d.setColor(new Color(random.nextInt(100), random.nextInt(100), random.nextInt(100)));
            
            // 随机位置和角度
            int x = 15 + i * 18;
            int y = 25 + random.nextInt(10);
            
            // 旋转文字
            double angle = (random.nextDouble() - 0.5) * 0.5;
            g2d.rotate(angle, x, y);
            g2d.drawString(String.valueOf(captchaText.charAt(i)), x, y);
            g2d.rotate(-angle, x, y);
        }
        
        g2d.dispose();
        
        // 转换为Base64
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ImageIO.write(image, "png", baos);
            byte[] imageBytes = baos.toByteArray();
            return Base64.getEncoder().encodeToString(imageBytes);
        } catch (IOException e) {
            throw new RuntimeException("生成验证码图片失败", e);
        }
    }
    
    /**
     * 清理过期验证码
     */
    private void cleanExpiredCaptcha() {
        long currentTime = System.currentTimeMillis();
        captchaStore.entrySet().removeIf(entry -> 
            currentTime - entry.getValue().getCreateTime() > CAPTCHA_EXPIRE_TIME
        );
    }
    
    /**
     * 验证码信息类
     */
    private static class CaptchaInfo {
        private final String text;
        private final long createTime;
        
        public CaptchaInfo(String text, long createTime) {
            this.text = text;
            this.createTime = createTime;
        }
        
        public String getText() {
            return text;
        }
        
        public long getCreateTime() {
            return createTime;
        }
    }
}