package com.likeusewin10.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.likeusewin10.dto.request.CommentRequest;
import com.likeusewin10.dto.response.CommentResponse;
import com.likeusewin10.dto.response.PageResult;
import com.likeusewin10.dto.response.Result;
import com.likeusewin10.entity.Comment;
import com.likeusewin10.entity.User;
import com.likeusewin10.mapper.CommentMapper;
import com.likeusewin10.mapper.UserMapper;
import com.likeusewin10.service.CommentService;
import com.likeusewin10.service.LikeService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

@Service
public class CommentServiceImpl implements CommentService {
    
    @Autowired
    private CommentMapper commentMapper;
    
    @Autowired
    private UserMapper userMapper;
    
    @Autowired
    private LikeService likeService;
    
    @Override
    public Result<PageResult<CommentResponse>> getCommentsByArticleId(Long articleId, Integer current, Integer size) {
        try {
            Page<Comment> page = new Page<>(current, size);
            QueryWrapper<Comment> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("article_id", articleId);
            queryWrapper.orderByDesc("created_at");
            
            Page<Comment> commentPage = commentMapper.selectPage(page, queryWrapper);
            
            List<CommentResponse> commentResponses = commentPage.getRecords().stream()
                    .map(this::convertToResponse)
                    .collect(Collectors.toList());
            
            PageResult<CommentResponse> pageResult = new PageResult<>();
            pageResult.setRecords(commentResponses);
            pageResult.setTotal(commentPage.getTotal());
            pageResult.setCurrent(commentPage.getCurrent());
            pageResult.setSize(commentPage.getSize());
            pageResult.setPages(commentPage.getPages());
            
            return Result.success(pageResult);
        } catch (Exception e) {
            return Result.error("获取评论列表失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> createComment(Long userId, CommentRequest request) {
        try {
            Comment comment = new Comment();
            BeanUtils.copyProperties(request, comment);
            comment.setUserId(userId);
            comment.setCreatedAt(LocalDateTime.now());
            comment.setUpdatedAt(LocalDateTime.now());
            
            commentMapper.insert(comment);
            return Result.success();
        } catch (Exception e) {
            return Result.error("创建评论失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> deleteComment(Long commentId, Long userId) {
        try {
            Comment comment = commentMapper.selectById(commentId);
            if (comment == null) {
                return Result.error("评论不存在");
            }
            
            // 验证评论作者
            if (!comment.getUserId().equals(userId)) {
                return Result.error("只能删除自己的评论");
            }
            
            commentMapper.deleteById(commentId);
            return Result.success();
            
        } catch (Exception e) {
            return Result.error("删除评论失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<PageResult<CommentResponse>> getUserComments(Long userId, Integer current, Integer size) {
        try {
            Page<Comment> page = new Page<>(current, size);
            QueryWrapper<Comment> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("user_id", userId)
                       .orderByDesc("created_at");
            
            Page<Comment> commentPage = commentMapper.selectPage(page, queryWrapper);
            
            List<CommentResponse> responseList = commentPage.getRecords().stream()
                .map(this::convertToResponse)
                .collect(Collectors.toList());
            
            PageResult<CommentResponse> pageResult = new PageResult<>();
            pageResult.setRecords(responseList);
            pageResult.setTotal(commentPage.getTotal());
            pageResult.setCurrent(commentPage.getCurrent());
            pageResult.setSize(commentPage.getSize());
            
            return Result.success(pageResult);
            
        } catch (Exception e) {
            return Result.error("获取用户评论失败：" + e.getMessage());
        }
    }
    
    private CommentResponse convertToResponse(Comment comment) {
        CommentResponse response = new CommentResponse();
        BeanUtils.copyProperties(comment, response);
        
        // 获取用户信息
        User user = userMapper.selectById(comment.getUserId());
        if (user != null) {
            response.setUserName(user.getUsername());
            response.setUserAvatar(user.getAvatarUrl());
        }
        
        // 获取点赞数量
        Integer likeCount = likeService.getLikeCount(comment.getId());
        response.setLikeCount(likeCount != null ? likeCount : 0);
        
        return response;
    }
}