package com.likeusewin10.service.impl;

import com.likeusewin10.dto.response.Result;
import com.likeusewin10.service.EmailService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Service;

import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * 邮箱服务实现类
 * 支持真实邮件发送功能
 * 
 * @author likeusewin10
 * @since 2025-01-10
 */
@Service
public class EmailServiceImpl implements EmailService {
    
    @Autowired
    private JavaMailSender mailSender;
    
    @Value("${spring.mail.username}")
    private String fromEmail;
    
    // 使用内存存储验证码，实际项目中应该使用Redis
    private final Map<String, String> verificationCodes = new ConcurrentHashMap<>();
    private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1);
    
    @Override
    public Result<Void> sendVerificationCode(String email, String code) {
        try {
            // 存储验证码，5分钟后过期
            verificationCodes.put(email, code);
            
            // 5分钟后删除验证码
            scheduler.schedule(() -> {
                verificationCodes.remove(email);
            }, 5, TimeUnit.MINUTES);
            
            // 发送真实邮件
            if (isRealEmailConfigured()) {
                sendRealEmail(email, code);
            } else {
                // 如果邮件配置不完整，则模拟发送
                System.out.println("邮件配置不完整，模拟发送验证码到邮箱: " + email + ", 验证码: " + code);
            }
            
            return Result.success();
            
        } catch (Exception e) {
            return Result.error("发送验证码失败：" + e.getMessage());
        }
    }
    
    /**
     * 检查邮件配置是否完整
     */
    private boolean isRealEmailConfigured() {
        return fromEmail != null && !fromEmail.equals("your-email@qq.com") && !fromEmail.isEmpty();
    }
    
    /**
     * 发送真实邮件
     */
    private void sendRealEmail(String toEmail, String code) {
        try {
            SimpleMailMessage message = new SimpleMailMessage();
            message.setFrom(fromEmail);
            message.setTo(toEmail);
            message.setSubject("个人博客系统 - 验证码");
            message.setText("您的验证码是: " + code + "\n\n验证码有效期为5分钟，请及时使用。\n\n如果这不是您的操作，请忽略此邮件。\n\n个人博客系统");
            
            mailSender.send(message);
            System.out.println("验证码邮件已发送到: " + toEmail);
            
        } catch (Exception e) {
            System.err.println("发送邮件失败: " + e.getMessage());
            // 邮件发送失败时，仍然在控制台显示验证码作为备用方案
            System.out.println("备用方案 - 验证码: " + code + " (邮箱: " + toEmail + ")");
        }
    }
    
    @Override
    public Result<Boolean> verifyCode(String email, String code) {
        try {
            String storedCode = verificationCodes.get(email);
            if (storedCode == null) {
                return Result.error("验证码已过期或不存在");
            }
            
            if (storedCode.equals(code)) {
                // 验证成功后删除验证码
                verificationCodes.remove(email);
                return Result.success(true);
            } else {
                return Result.error("验证码错误");
            }
            
        } catch (Exception e) {
            return Result.error("验证失败：" + e.getMessage());
        }
    }
    
    @Override
    public String generateVerificationCode() {
        Random random = new Random();
        StringBuilder code = new StringBuilder();
        for (int i = 0; i < 6; i++) {
            code.append(random.nextInt(10));
        }
        return code.toString();
    }
}