package com.likeusewin10.service.impl;

import com.likeusewin10.dto.request.LoginRequest;
import com.likeusewin10.dto.request.RegisterRequest;
import com.likeusewin10.dto.request.ForgotPasswordRequest;
import com.likeusewin10.dto.request.VerifyCodeRequest;
import com.likeusewin10.dto.request.ResetPasswordRequest;
import com.likeusewin10.dto.request.UpdatePasswordRequest;
import com.likeusewin10.dto.response.Result;
import com.likeusewin10.dto.response.UserResponse;
import com.likeusewin10.dto.response.LoginResponse;
import com.likeusewin10.entity.User;
import com.likeusewin10.mapper.UserMapper;
import com.likeusewin10.service.UserService;
import com.likeusewin10.service.EmailService;
import com.likeusewin10.service.CaptchaService;
import com.likeusewin10.util.JwtUtil;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.DigestUtils;

import javax.servlet.http.HttpServletRequest;

import java.time.LocalDateTime;

@Service
public class UserServiceImpl implements UserService {
    
    @Autowired
    private UserMapper userMapper;
    
    @Autowired
    private EmailService emailService;
    
    @Autowired
    private CaptchaService captchaService;
    
    @Override
    public Result<LoginResponse> login(LoginRequest request, HttpServletRequest httpRequest) {
        try {
            // 验证验证码
            if (request.getCaptcha() == null || request.getCaptcha().trim().isEmpty()) {
                return Result.error("请输入验证码");
            }
            if (request.getCaptchaId() == null || request.getCaptchaId().trim().isEmpty()) {
                return Result.error("验证码ID不能为空");
            }
            if (!captchaService.verifyCaptcha(request.getCaptchaId(), request.getCaptcha(), httpRequest)) {
                return Result.error("验证码错误或已过期");
            }
            
            // 查找用户
            User user = userMapper.findByUsername(request.getUsername());
            if (user == null) {
                return Result.error("用户不存在");
            }
            
            // 验证密码
            String encryptedPassword = DigestUtils.md5DigestAsHex(request.getPassword().getBytes());
            if (!encryptedPassword.equals(user.getPassword())) {
                return Result.error("密码错误");
            }
            
            // 检查用户状态
            if (user.getStatus() != 1) {
                return Result.error("用户已被禁用");
            }
            
            // 生成JWT token
            String token = JwtUtil.generateToken(user.getId(), user.getUsername());
            
            // 构建用户响应对象
            UserResponse userResponse = new UserResponse();
            BeanUtils.copyProperties(user, userResponse);
            
            // 构建登录响应对象
            LoginResponse loginResponse = new LoginResponse(token, userResponse);
            
            return Result.success("登录成功", loginResponse);
            
        } catch (Exception e) {
            return Result.error("登录失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> register(RegisterRequest request, HttpServletRequest httpRequest) {
        try {
            // 验证验证码
            if (request.getCaptcha() == null || request.getCaptcha().trim().isEmpty()) {
                return Result.error("请输入验证码");
            }
            if (request.getCaptchaId() == null || request.getCaptchaId().trim().isEmpty()) {
                return Result.error("验证码ID不能为空");
            }
            if (!captchaService.verifyCaptcha(request.getCaptchaId(), request.getCaptcha(), httpRequest)) {
                return Result.error("验证码错误或已过期");
            }
            
            // 检查用户名是否存在
            if (userMapper.findByUsername(request.getUsername()) != null) {
                return Result.error("用户名已存在");
            }
            
            // 检查邮箱是否存在
            if (userMapper.findByEmail(request.getEmail()) != null) {
                return Result.error("邮箱已被注册");
            }
            
            // 创建新用户
            User user = new User();
            user.setUsername(request.getUsername());
            user.setNickname(request.getNickname());
            user.setEmail(request.getEmail());
            user.setPassword(DigestUtils.md5DigestAsHex(request.getPassword().getBytes()));
            user.setRole("user");
            user.setStatus(1);
            user.setCreatedAt(LocalDateTime.now());
            user.setUpdatedAt(LocalDateTime.now());
            
            userMapper.insert(user);
            return Result.success();
            
        } catch (Exception e) {
            return Result.error("注册失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<UserResponse> getUserInfo(Long userId) {
        try {
            User user = userMapper.selectById(userId);
            if (user == null) {
                return Result.error("用户不存在");
            }
            
            UserResponse response = new UserResponse();
            BeanUtils.copyProperties(user, response);
            response.setCreatedAt(user.getCreatedAt());
            
            return Result.success(response);
            
        } catch (Exception e) {
            return Result.error("获取用户信息失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> updateUserInfo(Long userId, UserResponse userInfo) {
        try {
            User user = userMapper.selectById(userId);
            if (user == null) {
                return Result.error("用户不存在");
            }
            
            user.setNickname(userInfo.getNickname());
            user.setEmail(userInfo.getEmail());
            user.setAvatarUrl(userInfo.getAvatarUrl());
            user.setUpdatedAt(LocalDateTime.now());
            
            userMapper.updateById(user);
            return Result.success();
            
        } catch (Exception e) {
            return Result.error("更新用户信息失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> sendForgotPasswordCode(ForgotPasswordRequest request) {
        try {
            // 检查邮箱是否存在
            User user = userMapper.findByEmail(request.getEmail());
            if (user == null) {
                return Result.error("该邮箱未注册");
            }
            
            // 生成验证码
            String code = emailService.generateVerificationCode();
            
            // 发送验证码
            Result<Void> sendResult = emailService.sendVerificationCode(request.getEmail(), code);
            if (!sendResult.isSuccess()) {
                return sendResult;
            }
            
            return Result.success();
            
        } catch (Exception e) {
            return Result.error("发送验证码失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> verifyEmailCode(VerifyCodeRequest request) {
        try {
            // 验证邮箱验证码
            Result<Boolean> verifyResult = emailService.verifyCode(request.getEmail(), request.getVerificationCode());
            if (!verifyResult.isSuccess()) {
                return Result.error(verifyResult.getMessage());
            }
            
            if (!verifyResult.getData()) {
                return Result.error("验证码错误");
            }
            
            return Result.success();
            
        } catch (Exception e) {
            return Result.error("验证失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> resetPassword(ResetPasswordRequest request) {
        try {
            // 再次验证验证码
            Result<Boolean> verifyResult = emailService.verifyCode(request.getEmail(), request.getVerificationCode());
            if (!verifyResult.isSuccess()) {
                return Result.error("验证码已过期，请重新获取");
            }
            
            if (!verifyResult.getData()) {
                return Result.error("验证码错误");
            }
            
            // 查找用户
            User user = userMapper.findByEmail(request.getEmail());
            if (user == null) {
                return Result.error("用户不存在");
            }
            
            // 更新密码
            user.setPassword(DigestUtils.md5DigestAsHex(request.getNewPassword().getBytes()));
            user.setUpdatedAt(LocalDateTime.now());
            userMapper.updateById(user);
            
            return Result.success();
            
        } catch (Exception e) {
            return Result.error("重置密码失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Void> updatePassword(Long userId, UpdatePasswordRequest request, HttpServletRequest httpRequest) {
        try {
            // 验证验证码
            if (request.getCaptcha() == null || request.getCaptcha().trim().isEmpty()) {
                return Result.error("请输入验证码");
            }
            if (request.getCaptchaId() == null || request.getCaptchaId().trim().isEmpty()) {
                return Result.error("验证码ID不能为空");
            }
            if (!captchaService.verifyCaptcha(request.getCaptchaId(), request.getCaptcha(), httpRequest)) {
                return Result.error("验证码错误或已过期");
            }
            
            // 查找用户
            User user = userMapper.selectById(userId);
            if (user == null) {
                return Result.error("用户不存在");
            }
            
            // 验证当前密码
            String encryptedCurrentPassword = DigestUtils.md5DigestAsHex(request.getCurrentPassword().getBytes());
            if (!encryptedCurrentPassword.equals(user.getPassword())) {
                return Result.error("当前密码错误");
            }
            
            // 更新密码
            user.setPassword(DigestUtils.md5DigestAsHex(request.getNewPassword().getBytes()));
            user.setUpdatedAt(LocalDateTime.now());
            userMapper.updateById(user);
            
            return Result.success();
            
        } catch (Exception e) {
            return Result.error("修改密码失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Boolean> checkUsernameExists(String username) {
        try {
            User user = userMapper.findByUsername(username);
            return Result.success(user != null);
        } catch (Exception e) {
            return Result.error("检查用户名失败：" + e.getMessage());
        }
    }
    
    @Override
    public Result<Boolean> checkEmailExists(String email) {
        try {
            User user = userMapper.findByEmail(email);
            return Result.success(user != null);
        } catch (Exception e) {
            return Result.error("检查邮箱失败：" + e.getMessage());
        }
    }
}