-- 个人博客系统数据库建表脚本
-- 创建数据库（如果不存在）
CREATE DATABASE IF NOT EXISTS `musicgame_db` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- 创建用户（如果不存在）
CREATE USER IF NOT EXISTS 'musicgame_user'@'%' IDENTIFIED BY 'musicgame123';

-- 授权用户访问数据库
GRANT ALL PRIVILEGES ON `musicgame_db`.* TO 'musicgame_user'@'%';
FLUSH PRIVILEGES;

USE `musicgame_db`;

-- 用户表
CREATE TABLE IF NOT EXISTS `users` (
  `id` bigint(20) NOT NULL AUTO_INCREMENT COMMENT '用户ID',
  `username` varchar(50) NOT NULL COMMENT '用户名',
  `email` varchar(100) NOT NULL COMMENT '邮箱',
  `password` varchar(255) NOT NULL COMMENT '密码（加密后）',
  `nickname` varchar(50) DEFAULT NULL COMMENT '昵称',
  `avatar` varchar(255) DEFAULT NULL COMMENT '头像URL',
  `bio` text COMMENT '个人简介',
  `role` varchar(20) DEFAULT 'user' COMMENT '角色（admin/user）',
  `status` tinyint(1) DEFAULT 1 COMMENT '状态（1:正常 0:禁用）',
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `updated_at` datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_username` (`username`),
  UNIQUE KEY `uk_email` (`email`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='用户表';

-- 游戏表
CREATE TABLE IF NOT EXISTS `games` (
  `id` bigint(20) NOT NULL AUTO_INCREMENT COMMENT '游戏ID',
  `title` varchar(200) NOT NULL COMMENT '游戏标题',
  `description` text COMMENT '游戏描述',
  `category` varchar(50) NOT NULL COMMENT '游戏分类',
  `video_url` varchar(500) NOT NULL COMMENT '视频URL',
  `thumbnail` varchar(500) DEFAULT NULL COMMENT '缩略图URL',
  `duration` int(11) DEFAULT NULL COMMENT '视频时长（秒）',
  `views` int(11) DEFAULT 0 COMMENT '观看次数',
  `likes` int(11) DEFAULT 0 COMMENT '点赞数',
  `status` tinyint(1) DEFAULT 1 COMMENT '状态（1:正常 0:下架）',
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `updated_at` datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`id`),
  KEY `idx_category` (`category`),
  KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='游戏表';

-- 音乐表
CREATE TABLE IF NOT EXISTS `music` (
  `id` bigint(20) NOT NULL AUTO_INCREMENT COMMENT '音乐ID',
  `title` varchar(200) NOT NULL COMMENT '音乐标题',
  `artist` varchar(100) NOT NULL COMMENT '艺术家',
  `album` varchar(100) DEFAULT NULL COMMENT '专辑',
  `genre` varchar(50) DEFAULT NULL COMMENT '音乐类型',
  `audio_url` varchar(500) NOT NULL COMMENT '音频URL',
  `cover_image` varchar(500) DEFAULT NULL COMMENT '封面图片URL',
  `duration` int(11) DEFAULT NULL COMMENT '音乐时长（秒）',
  `play_count` int(11) DEFAULT 0 COMMENT '播放次数',
  `likes` int(11) DEFAULT 0 COMMENT '点赞数',
  `status` tinyint(1) DEFAULT 1 COMMENT '状态（1:正常 0:下架）',
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `updated_at` datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`id`),
  KEY `idx_genre` (`genre`),
  KEY `idx_artist` (`artist`),
  KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='音乐表';

-- 文章表
CREATE TABLE IF NOT EXISTS `articles` (
  `id` bigint(20) NOT NULL AUTO_INCREMENT COMMENT '文章ID',
  `title` varchar(200) NOT NULL COMMENT '文章标题',
  `content` longtext NOT NULL COMMENT '文章内容',
  `summary` text COMMENT '文章摘要',
  `category` varchar(50) NOT NULL COMMENT '文章分类',
  `tags` varchar(500) DEFAULT NULL COMMENT '标签（逗号分隔）',
  `author_id` bigint(20) NOT NULL COMMENT '作者ID',
  `cover_image` varchar(500) DEFAULT NULL COMMENT '封面图片URL',
  `views` int(11) DEFAULT 0 COMMENT '浏览次数',
  `likes` int(11) DEFAULT 0 COMMENT '点赞数',
  `status` tinyint(1) DEFAULT 1 COMMENT '状态（1:已发布 0:草稿）',
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `updated_at` datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`id`),
  KEY `idx_author_id` (`author_id`),
  KEY `idx_category` (`category`),
  KEY `idx_status` (`status`),
  CONSTRAINT `fk_articles_author` FOREIGN KEY (`author_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='文章表';

-- 评论表
CREATE TABLE IF NOT EXISTS `comments` (
  `id` bigint(20) NOT NULL AUTO_INCREMENT COMMENT '评论ID',
  `article_id` bigint(20) NOT NULL COMMENT '文章ID',
  `user_id` bigint(20) NOT NULL COMMENT '用户ID',
  `parent_id` bigint(20) DEFAULT NULL COMMENT '父评论ID（用于回复）',
  `content` text NOT NULL COMMENT '评论内容',
  `image_urls` json DEFAULT NULL COMMENT '评论图片URL数组',
  `likes` int(11) DEFAULT 0 COMMENT '点赞数',
  `status` tinyint(1) DEFAULT 1 COMMENT '状态（1:正常 0:删除）',
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `updated_at` datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`id`),
  KEY `idx_article_id` (`article_id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_parent_id` (`parent_id`),
  CONSTRAINT `fk_comments_article` FOREIGN KEY (`article_id`) REFERENCES `articles` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_comments_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_comments_parent` FOREIGN KEY (`parent_id`) REFERENCES `comments` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='评论表';
