// 评论功能测试脚本
const API_BASE_URL = 'http://localhost:8080/api';

// 动态导入 node-fetch
let fetch;

if (typeof window === 'undefined') {
  // Node.js 环境
  import('node-fetch').then(nodeFetch => {
    fetch = nodeFetch.default;
    testCommentAPIs();
  }).catch(error => {
    console.error('导入 node-fetch 失败:', error);
  });
} else {
  // 浏览器环境
  testCommentAPIs();
}

// 测试用的登录token（需要先登录获取）
let authToken = '';

// 登录获取token
async function login() {
  try {
    const response = await fetch(`${API_BASE_URL}/user/login`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        username: 'testuser',
        password: '123456'
      })
    });
    
    const result = await response.json();
    if (result.success && result.data && result.data.token) {
      authToken = result.data.token;
      console.log('✅ 登录成功，获取到token');
      return true;
    } else {
      console.log('❌ 登录失败:', result.message);
      return false;
    }
  } catch (error) {
    console.log('❌ 登录请求失败:', error.message);
    return false;
  }
}

// 测试获取文章评论
async function testGetComments(articleId = 1) {
  try {
    const response = await fetch(`${API_BASE_URL}/comment/article/${articleId}?current=1&size=10`);
    const result = await response.json();
    
    if (response.ok) {
      console.log('✅ 获取文章评论 - 正常');
      console.log('   完整响应:', JSON.stringify(result, null, 2));
      console.log('   评论数量:', result.data?.records?.length || 0);
      if (result.data?.total) {
        console.log('   总评论数:', result.data.total);
      }
      return result.data?.records || [];
    } else {
      console.log('❌ 获取文章评论 - 失败:', result.message || result);
      console.log('   响应状态:', response.status);
      console.log('   完整响应:', JSON.stringify(result, null, 2));
      return [];
    }
  } catch (error) {
    console.log('❌ 获取文章评论 - 错误:', error.message);
    return [];
  }
}

// 测试创建评论
async function testCreateComment(articleId = 1) {
  if (!authToken) {
    console.log('❌ 创建评论 - 需要先登录');
    return null;
  }
  
  try {
    const response = await fetch(`${API_BASE_URL}/comment`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${authToken}`
      },
      body: JSON.stringify({
        articleId: articleId,
        content: '这是一条测试评论，时间：' + new Date().toLocaleString()
      })
    });
    
    const result = await response.json();
    
    if (response.ok && (result.success || result.code === 200)) {
      console.log('✅ 创建评论 - 正常');
      return true;
    } else {
      console.log('❌ 创建评论 - 失败:', result.message || result);
      console.log('   响应状态:', response.status);
      console.log('   完整响应:', JSON.stringify(result, null, 2));
      return false;
    }
  } catch (error) {
    console.log('❌ 创建评论 - 错误:', error.message);
    return false;
  }
}

// 测试删除评论
async function testDeleteComment(commentId) {
  if (!authToken) {
    console.log('❌ 删除评论 - 需要先登录');
    return false;
  }
  
  if (!commentId) {
    console.log('❌ 删除评论 - 没有可删除的评论ID');
    return false;
  }
  
  try {
    const response = await fetch(`${API_BASE_URL}/comment/${commentId}`, {
      method: 'DELETE',
      headers: {
        'Authorization': `Bearer ${authToken}`
      }
    });
    
    const result = await response.json();
    
    if (response.ok && (result.success || result.code === 200)) {
      console.log('✅ 删除评论 - 正常');
      return true;
    } else {
      console.log('❌ 删除评论 - 失败:', result.message);
      return false;
    }
  } catch (error) {
    console.log('❌ 删除评论 - 错误:', error.message);
    return false;
  }
}

// 主测试函数
async function testCommentAPIs() {
  console.log('\n=== 评论功能测试开始 ===\n');
  
  // 1. 先登录
  const loginSuccess = await login();
  if (!loginSuccess) {
    console.log('\n❌ 无法继续测试，登录失败');
    return;
  }
  
  // 2. 获取现有评论
  console.log('\n--- 测试获取评论 ---');
  const existingComments = await testGetComments(1);
  
  // 3. 创建新评论
  console.log('\n--- 测试创建评论 ---');
  const createSuccess = await testCreateComment(1);
  
  // 4. 再次获取评论，验证创建是否成功
  if (createSuccess) {
    console.log('\n--- 验证评论创建结果 ---');
    const newComments = await testGetComments(1);
    if (newComments.length > existingComments.length) {
      console.log('✅ 评论创建验证 - 成功，评论数量增加了');
      
      // 5. 测试删除刚创建的评论
      const latestComment = newComments[0]; // 最新的评论应该在第一个
      if (latestComment && latestComment.id) {
        console.log('\n--- 测试删除评论 ---');
        await testDeleteComment(latestComment.id);
        
        // 6. 验证删除结果
        console.log('\n--- 验证评论删除结果 ---');
        const finalComments = await testGetComments(1);
        if (finalComments.length === existingComments.length) {
          console.log('✅ 评论删除验证 - 成功，评论数量恢复原状');
        } else {
          console.log('❌ 评论删除验证 - 失败，评论数量不匹配');
        }
      }
    } else {
      console.log('❌ 评论创建验证 - 失败，评论数量没有增加');
    }
  }
  
  console.log('\n=== 评论功能测试结束 ===\n');
}

// 如果直接运行此脚本
if (typeof window === 'undefined' && typeof module !== 'undefined') {
  // Node.js 环境下的导出
  module.exports = { testCommentAPIs };
}