#!/bin/bash

# 个人博客项目部署脚本
# 作者: likeusewin10
# 创建时间: 2025-01-10

set -e  # 遇到错误立即退出

echo "========================================"
echo "个人博客项目自动化部署脚本"
echo "========================================"

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 项目配置
PROJECT_NAME="personal-blog"
BACKEND_DIR="./backend"
FRONTEND_DIR="./Vue3"
DOCKER_COMPOSE_FILE="./docker-compose.yml"

# 函数：打印带颜色的消息
print_message() {
    echo -e "${2}[$(date +'%Y-%m-%d %H:%M:%S')] $1${NC}"
}

# 函数：检查命令是否存在
check_command() {
    if ! command -v $1 &> /dev/null; then
        print_message "错误: $1 命令未找到，请先安装 $1" $RED
        exit 1
    fi
}

# 函数：构建前端
build_frontend() {
    print_message "开始构建前端项目..." $BLUE
    
    cd $FRONTEND_DIR
    
    # 检查 package.json 是否存在
    if [ ! -f "package.json" ]; then
        print_message "错误: 未找到 package.json 文件" $RED
        exit 1
    fi
    
    # 安装依赖
    print_message "安装前端依赖..." $YELLOW
    npm install
    
    # 构建项目
    print_message "构建前端项目..." $YELLOW
    npm run build
    
    # 检查构建结果
    if [ ! -d "dist" ]; then
        print_message "错误: 前端构建失败，未生成 dist 目录" $RED
        exit 1
    fi
    
    print_message "前端构建完成" $GREEN
    cd ..
}

# 函数：构建后端
build_backend() {
    print_message "开始构建后端项目..." $BLUE
    
    cd $BACKEND_DIR
    
    # 检查 pom.xml 是否存在
    if [ ! -f "pom.xml" ]; then
        print_message "错误: 未找到 pom.xml 文件" $RED
        exit 1
    fi
    
    # 清理并打包
    print_message "清理并打包后端项目..." $YELLOW
    mvn clean package -DskipTests
    
    # 检查构建结果
    if [ ! -f "target/personal-blog-1.0.0.jar" ]; then
        print_message "错误: 后端构建失败，未生成 jar 文件" $RED
        exit 1
    fi
    
    print_message "后端构建完成" $GREEN
    cd ..
}

# 函数：准备部署环境
prepare_deployment() {
    print_message "准备部署环境..." $BLUE
    
    # 创建必要的目录
    mkdir -p uploads logs
    
    # 设置权限
    chmod 755 uploads logs
    
    # 检查 Docker 数据卷是否存在
    if ! docker volume ls | grep -q "musicgame-mysql-data"; then
        print_message "创建 MySQL 数据卷..." $YELLOW
        docker volume create musicgame-mysql-data
    fi
    
    print_message "部署环境准备完成" $GREEN
}

# 函数：部署应用
deploy_application() {
    print_message "开始部署应用..." $BLUE
    
    # 停止现有服务
    print_message "停止现有服务..." $YELLOW
    docker-compose -f $DOCKER_COMPOSE_FILE down
    
    # 构建并启动服务
    print_message "构建并启动服务..." $YELLOW
    docker-compose -f $DOCKER_COMPOSE_FILE up -d --build
    
    # 等待服务启动
    print_message "等待服务启动..." $YELLOW
    sleep 30
    
    # 检查服务状态
    print_message "检查服务状态..." $YELLOW
    docker-compose -f $DOCKER_COMPOSE_FILE ps
    
    print_message "应用部署完成" $GREEN
}

# 函数：健康检查
health_check() {
    print_message "执行健康检查..." $BLUE
    
    # 检查后端服务
    print_message "检查后端服务..." $YELLOW
    for i in {1..10}; do
        if curl -f http://localhost:8081/api/health &> /dev/null; then
            print_message "后端服务健康检查通过" $GREEN
            break
        else
            if [ $i -eq 10 ]; then
                print_message "警告: 后端服务健康检查失败" $YELLOW
            else
                print_message "等待后端服务启动... ($i/10)" $YELLOW
                sleep 10
            fi
        fi
    done
    
    # 检查前端服务
    print_message "检查前端服务..." $YELLOW
    if curl -f http://localhost &> /dev/null; then
        print_message "前端服务健康检查通过" $GREEN
    else
        print_message "警告: 前端服务健康检查失败" $YELLOW
    fi
    
    # 检查数据库服务
    print_message "检查数据库服务..." $YELLOW
    if docker exec personal-blog-mysql mysqladmin ping -h localhost -u root -proot123456 &> /dev/null; then
        print_message "数据库服务健康检查通过" $GREEN
    else
        print_message "警告: 数据库服务健康检查失败" $YELLOW
    fi
}

# 函数：显示部署信息
show_deployment_info() {
    print_message "部署完成！" $GREEN
    echo ""
    echo "========================================"
    echo "部署信息:"
    echo "========================================"
    echo "前端地址: http://likeusewin10.xyz"
    echo "后端API: http://likeusewin10.xyz/api"
    echo "数据库: MySQL 8.0.36 (Docker)"
    echo ""
    echo "服务状态:"
    docker-compose -f $DOCKER_COMPOSE_FILE ps
    echo ""
    echo "查看日志命令:"
    echo "  前端: docker logs personal-blog-frontend"
    echo "  后端: docker logs personal-blog-backend"
    echo "  数据库: docker logs personal-blog-mysql"
    echo ""
    echo "停止服务: docker-compose down"
    echo "重启服务: docker-compose restart"
    echo "========================================"
}

# 主函数
main() {
    # 检查必要的命令
    check_command "node"
    check_command "npm"
    check_command "mvn"
    check_command "docker"
    check_command "docker-compose"
    
    # 执行部署步骤
    build_frontend
    build_backend
    prepare_deployment
    deploy_application
    health_check
    show_deployment_info
    
    print_message "部署脚本执行完成！" $GREEN
}

# 脚本入口
if [ "$1" = "--help" ] || [ "$1" = "-h" ]; then
    echo "个人博客项目部署脚本"
    echo ""
    echo "用法: $0 [选项]"
    echo ""
    echo "选项:"
    echo "  --help, -h     显示帮助信息"
    echo "  --frontend     仅构建前端"
    echo "  --backend      仅构建后端"
    echo "  --deploy       仅部署（不构建）"
    echo ""
    exit 0
elif [ "$1" = "--frontend" ]; then
    build_frontend
elif [ "$1" = "--backend" ]; then
    build_backend
elif [ "$1" = "--deploy" ]; then
    prepare_deployment
    deploy_application
    health_check
    show_deployment_info
else
    main
fi