const axios = require('axios');

// 配置基础URL
const BASE_URL = 'http://localhost:8080';

// 创建axios实例
const api = axios.create({
  baseURL: BASE_URL,
  timeout: 10000,
  headers: {
    'Content-Type': 'application/json'
  }
});

// 测试数据
const testData = {
  username: 'testuser',
  password: '123456',
  commentId: 1 // 测试评论ID
};

// 登录函数
async function login() {
  try {
    const response = await api.post('/api/user/login', {
      username: testData.username,
      password: testData.password
    });
    
    if (response.data.success) {
      const token = response.data.data.token;
      // 设置token到请求头
      api.defaults.headers.common['Authorization'] = `Bearer ${token}`;
      console.log('✅ 登录成功');
      return true;
    } else {
      console.log('❌ 登录失败:', response.data.message);
      return false;
    }
  } catch (error) {
    console.log('❌ 登录请求失败:', error.message);
    return false;
  }
}

// 测试点赞功能
async function testLikeFeature() {
  console.log('\n=== 点赞功能测试开始 ===\n');
  
  try {
    // 1. 获取初始点赞状态
    console.log('--- 获取初始点赞状态 ---');
    const initialStatus = await api.get(`/api/like/count/${testData.commentId}`);
    console.log('✅ 获取点赞数量 - 正常');
    console.log('   初始点赞数:', initialStatus.data.data.likeCount);
    const initialCount = initialStatus.data.data.likeCount;
    
    // 2. 检查是否已点赞
    const checkStatus = await api.get(`/api/like/check/${testData.commentId}`);
    console.log('✅ 检查点赞状态 - 正常');
    console.log('   是否已点赞:', checkStatus.data.data.hasLiked);
    
    // 3. 点赞操作
    console.log('\n--- 测试点赞操作 ---');
    const likeResponse = await api.post(`/api/like/toggle/${testData.commentId}`);
    console.log('✅ 点赞操作 - 正常');
    console.log('   操作结果:', likeResponse.data.data.message);
    console.log('   点赞数量:', likeResponse.data.data.likeCount);
    console.log('   点赞状态:', likeResponse.data.data.hasLiked);
    
    // 4. 验证点赞结果
    console.log('\n--- 验证点赞结果 ---');
    const afterLikeStatus = await api.get(`/api/like/count/${testData.commentId}`);
    const newCount = afterLikeStatus.data.data.likeCount;
    
    if (checkStatus.data.data.hasLiked) {
      // 如果之前已点赞，现在应该是取消点赞
      if (newCount === initialCount - 1) {
        console.log('✅ 取消点赞验证 - 成功，点赞数量减少了1');
      } else {
        console.log('❌ 取消点赞验证 - 失败，点赞数量未正确减少');
      }
    } else {
      // 如果之前未点赞，现在应该是点赞
      if (newCount === initialCount + 1) {
        console.log('✅ 点赞验证 - 成功，点赞数量增加了1');
      } else {
        console.log('❌ 点赞验证 - 失败，点赞数量未正确增加');
      }
    }
    
    // 5. 再次点赞/取消点赞
    console.log('\n--- 测试再次点赞/取消点赞 ---');
    const toggleResponse = await api.post(`/api/like/toggle/${testData.commentId}`);
    console.log('✅ 再次点赞操作 - 正常');
    console.log('   操作结果:', toggleResponse.data.data.message);
    console.log('   点赞数量:', toggleResponse.data.data.likeCount);
    console.log('   点赞状态:', toggleResponse.data.data.hasLiked);
    
    // 6. 验证恢复状态
    console.log('\n--- 验证状态恢复 ---');
    const finalStatus = await api.get(`/api/like/count/${testData.commentId}`);
    const finalCount = finalStatus.data.data.likeCount;
    
    if (finalCount === initialCount) {
      console.log('✅ 状态恢复验证 - 成功，点赞数量恢复到初始状态');
    } else {
      console.log('❌ 状态恢复验证 - 失败，点赞数量未恢复到初始状态');
    }
    
  } catch (error) {
    console.log('❌ 点赞功能测试失败:', error.response?.data?.message || error.message);
    if (error.response?.status) {
      console.log('   HTTP状态码:', error.response.status);
    }
  }
}

// 测试评论中的点赞数量显示
async function testCommentLikeDisplay() {
  console.log('\n--- 测试评论中的点赞数量显示 ---');
  
  try {
    const response = await api.get('/api/comment/article/1?current=1&size=10');
    
    if (response.data.success) {
      console.log('✅ 获取文章评论 - 正常');
      const comments = response.data.data.records;
      
      if (comments.length > 0) {
        console.log('   评论列表中的点赞数量:');
        comments.forEach((comment, index) => {
          console.log(`   评论${index + 1} (ID: ${comment.id}): ${comment.likeCount} 个点赞`);
        });
      } else {
        console.log('   暂无评论');
      }
    } else {
      console.log('❌ 获取文章评论失败:', response.data.message);
    }
  } catch (error) {
    console.log('❌ 获取评论失败:', error.response?.data?.message || error.message);
  }
}

// 主测试函数
async function runTests() {
  console.log('开始测试点赞功能...');
  
  // 登录
  const loginSuccess = await login();
  if (!loginSuccess) {
    console.log('登录失败，终止测试');
    return;
  }
  
  // 测试点赞功能
  await testLikeFeature();
  
  // 测试评论中的点赞数量显示
  await testCommentLikeDisplay();
  
  console.log('\n=== 点赞功能测试结束 ===\n');
}

// 运行测试
runTests().catch(console.error);