# 压缩包上传部署操作手册

## 📋 操作前准备

### 环境要求
- 本地开发环境（Node.js + Maven）
- 服务器 SSH 访问权限或宝塔面板访问
- 服务器已安装 Docker 和 Docker Compose
- 文件传输工具（SCP/SFTP/宝塔面板文件管理）

### 优势说明
- **简单直接** - 无需复杂的CI/CD配置，直接上传部署
- **环境可控** - 本地构建确保环境一致性
- **快速部署** - 压缩包上传后快速解压部署
- **灵活性强** - 支持任意服务器环境，不依赖特定平台
- **离线部署** - 支持内网环境和离线服务器部署

## 🚀 方案一：本地构建压缩包部署

### 步骤1：本地项目构建

在项目根目录执行以下命令：

```bash
# 进入项目根目录
cd /path/to/your/project

# 前端构建
cd Vue3
npm install
npm run build
cd ..

# 后端构建
cd backend
mvn clean package -DskipTests
cd ..
```

### 步骤2：创建部署包构建脚本

创建文件 `build-deploy-package.sh`，复制以下内容：

```bash
#!/bin/bash

echo "=== 开始构建部署包 ==="

# 检查构建产物
if [ ! -d "Vue3/dist" ]; then
    echo "错误：前端构建产物不存在，请先执行前端构建"
    exit 1
fi

if [ ! -f "backend/target/blog-backend-1.0.0.jar" ]; then
    echo "错误：后端构建产物不存在，请先执行后端构建"
    exit 1
fi

# 创建部署包目录
rm -rf deploy-package
mkdir -p deploy-package

# 复制前端构建产物
cp -r Vue3/dist deploy-package/frontend
echo "✓ 前端文件已复制"

# 复制后端JAR包
cp backend/target/blog-backend-1.0.0.jar deploy-package/backend.jar
echo "✓ 后端JAR包已复制"

# 复制配置文件
cp docker-compose.yml deploy-package/
cp -r nginx deploy-package/
cp -r sql deploy-package/
echo "✓ 配置文件已复制"

# 创建部署脚本
cat > deploy-package/deploy.sh << 'DEPLOY_EOF'
#!/bin/bash
echo "=== 开始部署 ==="

# 停止现有服务
docker-compose down

# 备份现有数据（如果存在）
if [ -d "uploads" ]; then
    cp -r uploads uploads_backup_$(date +%Y%m%d_%H%M%S)
    echo "✓ 已备份上传文件"
fi

# 创建必要目录
mkdir -p uploads/articles uploads/avatars uploads/comments

# 启动服务
docker-compose up -d

echo "=== 部署完成 ==="
echo "前端访问: http://your-aliyun-ip"
echo "后端API: http://your-aliyun-ip:8080"
DEPLOY_EOF

chmod +x deploy-package/deploy.sh

# 打包
tar -czf blog-deployment-$(date +%Y%m%d_%H%M%S).tar.gz deploy-package/
echo "✓ 部署包已创建: blog-deployment-$(date +%Y%m%d_%H%M%S).tar.gz"

echo "=== 构建完成 ==="
EOF

chmod +x build-deploy-package.sh

# Windows环境：创建部署包构建脚本
# 创建 build-deploy-package.bat (Windows批处理脚本)
cat > build-deploy-package.bat << 'EOF'
@echo off
chcp 65001 >nul

echo === 开始构建部署包 ===

REM 检查构建产物
if not exist "Vue3\dist" (
    echo 错误：前端构建产物不存在，请先执行前端构建
    pause
    exit /b 1
)

if not exist "backend\target\blog-backend-1.0.0.jar" (
    echo 错误：后端构建产物不存在，请先执行后端构建
    pause
    exit /b 1
)

REM 创建部署包目录
if exist "deploy-package" rmdir /s /q "deploy-package"
mkdir "deploy-package"

REM 复制前端构建产物
xcopy "Vue3\dist" "deploy-package\frontend" /e /i /h /y
echo ✓ 前端文件已复制

REM 复制后端JAR包
copy "backend\target\blog-backend-1.0.0.jar" "deploy-package\backend.jar"
echo ✓ 后端JAR包已复制

REM 复制配置文件
copy "docker-compose.yml" "deploy-package\"
xcopy "nginx" "deploy-package\nginx" /e /i /h /y
xcopy "sql" "deploy-package\sql" /e /i /h /y
echo ✓ 配置文件已复制

REM 创建Linux部署脚本（在阿里云服务器上执行）
echo #!/bin/bash > deploy-package\deploy.sh
echo echo "=== 开始部署 ===" >> deploy-package\deploy.sh
echo. >> deploy-package\deploy.sh
echo # 停止现有服务 >> deploy-package\deploy.sh
echo docker-compose down >> deploy-package\deploy.sh
echo. >> deploy-package\deploy.sh
echo # 备份现有数据（如果存在） >> deploy-package\deploy.sh
echo if [ -d "uploads" ]; then >> deploy-package\deploy.sh
echo     cp -r uploads uploads_backup_$(date +%%Y%%m%%d_%%H%%M%%S) >> deploy-package\deploy.sh
echo     echo "✓ 已备份上传文件" >> deploy-package\deploy.sh
echo fi >> deploy-package\deploy.sh
echo. >> deploy-package\deploy.sh
echo # 创建必要目录 >> deploy-package\deploy.sh
echo mkdir -p uploads/articles uploads/avatars uploads/comments >> deploy-package\deploy.sh
echo. >> deploy-package\deploy.sh
echo # 启动服务 >> deploy-package\deploy.sh
echo docker-compose up -d >> deploy-package\deploy.sh
echo. >> deploy-package\deploy.sh
echo echo "=== 部署完成 ===" >> deploy-package\deploy.sh
echo echo "前端访问: http://your-aliyun-ip" >> deploy-package\deploy.sh
echo echo "后端API: http://your-aliyun-ip:8080" >> deploy-package\deploy.sh

REM 使用PowerShell创建tar.gz压缩包
set timestamp=%date:~0,4%%date:~5,2%%date:~8,2%_%time:~0,2%%time:~3,2%%time:~6,2%
set timestamp=%timestamp: =0%
powershell -Command "Compress-Archive -Path 'deploy-package' -DestinationPath 'blog-deployment-%timestamp%.zip' -Force"
echo ✓ 部署包已创建: blog-deployment-%timestamp%.zip

echo === 构建完成 ===
echo 请将 blog-deployment-%timestamp%.zip 上传到阿里云服务器
pause
EOF
    
    # 创建部署包
    - name: 创建部署包
      run: |
        mkdir -p deploy-package/frontend
        mkdir -p deploy-package/backend
        mkdir -p deploy-package/scripts
        mkdir -p deploy-package/config
        
        # 复制前端构建产物
        cp -r Vue3/dist/* deploy-package/frontend/
        
        # 复制后端JAR文件
        cp backend/target/blog-backend-*.jar deploy-package/backend/
        
        # 复制配置文件
        cp backend/src/main/resources/application-prod.yml deploy-package/config/
        
        # 复制SQL文件
        cp -r sql deploy-package/
        
        # 创建部署脚本
        cat > deploy-package/scripts/deploy.sh << 'EOF'
        #!/bin/bash
        echo "开始部署博客项目..."
        
        # 停止现有服务
        sudo systemctl stop blog-backend 2>/dev/null || true
        
        # 备份现有文件
        if [ -d "/www/wwwroot/blog" ]; then
            sudo cp -r /www/wwwroot/blog /www/wwwroot/blog.backup.$(date +%Y%m%d_%H%M%S)
        fi
        
        # 创建目录结构
        sudo mkdir -p /www/wwwroot/blog/{frontend,backend,uploads/{articles,avatars,comments},logs}
        
        # 部署前端
        sudo cp -r frontend/* /www/wwwroot/blog/frontend/
        
        # 部署后端
        sudo cp backend/*.jar /www/wwwroot/blog/backend/
        sudo cp config/application-prod.yml /www/wwwroot/blog/backend/
        
        # 设置权限
        sudo chown -R www:www /www/wwwroot/blog/
        sudo chmod -R 755 /www/wwwroot/blog/
        
        # 启动服务
        sudo systemctl start blog-backend
        sudo systemctl enable blog-backend
        
        # 重载Nginx
        sudo nginx -s reload
        
        echo "部署完成！"
        EOF
        
        chmod +x deploy-package/scripts/deploy.sh
    
    # 打包部署文件
    - name: 打包部署文件
      run: |
        cd deploy-package
        tar -czf ../blog-deploy-$(date +%Y%m%d_%H%M%S).tar.gz .
        cd ..
        ls -la blog-deploy-*.tar.gz
    
    # 上传构建产物
    - name: 上传构建产物
      uses: actions/upload-artifact@v3
      with:
        name: blog-deploy-package
        path: blog-deploy-*.tar.gz
        retention-days: 30
    
    # 部署到服务器（可选）
    - name: 部署到服务器
      if: github.ref == 'refs/heads/main'
      run: |
        echo "准备部署到生产服务器..."
        # 这里可以添加 SSH 部署脚本
        # 需要在 GitHub Secrets 中配置服务器信息
```

### 步骤3：执行构建脚本

**Linux/Mac操作步骤：**
```bash
# 给脚本执行权限
chmod +x build-deploy-package.sh

# 执行构建脚本
./build-deploy-package.sh

# 查看生成的部署包
ls -la blog-deployment-*.tar.gz
```

**Windows操作步骤：**
```cmd
REM 在Windows命令提示符中执行
build-deploy-package.bat

REM 查看生成的部署包
dir blog-deployment-*.zip

REM 部署包内容预览（使用PowerShell）
powershell -Command "Get-ChildItem -Path 'deploy-package' -Recurse | Select-Object Name, Length"
```

**或者使用PowerShell：**
```powershell
# 在PowerShell中执行
.\build-deploy-package.bat

# 查看生成的部署包
Get-ChildItem blog-deployment-*.zip

# 部署包内容预览
Expand-Archive -Path (Get-ChildItem blog-deployment-*.zip | Select-Object -First 1).Name -DestinationPath temp-preview -Force
Get-ChildItem temp-preview -Recurse
Remove-Item temp-preview -Recurse -Force
```

### 步骤4：上传部署包到阿里云服务器

**方式一：使用WinSCP上传（推荐Windows用户）**
```cmd
REM 1. 下载安装WinSCP: https://winscp.net/
REM 2. 连接信息：
REM    主机名: your-aliyun-ip
REM    端口: 22
REM    用户名: root (或其他用户)
REM    密码: your-password
REM 3. 上传 blog-deployment-*.zip 到 /opt/blog/ 目录
```

**方式二：使用PowerShell的SCP（需要OpenSSH）**
```powershell
# 确保Windows已安装OpenSSH客户端
# 上传部署包到阿里云服务器
scp blog-deployment-*.zip root@your-aliyun-ip:/opt/blog/

# 登录阿里云服务器
ssh root@your-aliyun-ip
```

**方式三：使用宝塔面板上传**
```cmd
REM 1. 登录宝塔面板 http://your-aliyun-ip:8888
REM 2. 进入文件管理
REM 3. 导航到 /opt/blog 目录（如不存在则创建）
REM 4. 点击上传，选择 blog-deployment-*.zip 文件
REM 5. 等待上传完成
```

**方式四：使用阿里云控制台上传**
```cmd
REM 1. 登录阿里云控制台
REM 2. 进入ECS实例管理
REM 3. 使用"发送文件"功能上传部署包
REM 4. 或使用阿里云的远程连接功能
```

## 🚀 方案二：服务器端解压部署

### 步骤1：登录服务器解压部署包

```bash
# SSH登录服务器
ssh user@your-server

# 进入部署目录
cd /opt/blog

# 查看上传的部署包
ls -la blog-deployment-*.tar.gz

# 解压部署包
tar -xzf blog-deployment-*.tar.gz

# 查看解压内容
ls -la deploy-package/
```

### 步骤2：执行部署脚本

```bash
# 进入部署包目录
cd deploy-package

# 查看部署脚本内容
cat deploy.sh

# 执行部署脚本
./deploy.sh

# 查看服务状态
docker-compose ps
```

### 步骤3：验证部署结果

```bash
# 检查容器状态
docker-compose logs -f

# 测试前端访问
curl -I http://localhost

# 测试后端API
curl http://localhost:8080/api/health

# 查看端口占用
netstat -tlnp | grep -E ':(80|8080|3306)'
```

## 🚀 方案三：宝塔面板可视化部署

### 步骤1：通过宝塔面板上传

```bash
# 1. 登录宝塔面板
# 访问: http://your-server:8888
# 输入用户名和密码

# 2. 进入文件管理
# 点击左侧菜单 "文件"

# 3. 创建部署目录
# 导航到根目录，创建 /opt/blog 目录

# 4. 上传部署包
# 点击上传按钮，选择 blog-deployment-*.tar.gz
# 等待上传完成
```

### 步骤2：在线解压和部署

```bash
# 1. 在宝塔面板文件管理中
# 右键点击 blog-deployment-*.tar.gz
# 选择 "解压"

# 2. 进入解压后的目录
# 双击进入 deploy-package 目录

# 3. 查看文件结构
# frontend/     - 前端构建文件
# backend.jar   - 后端JAR包
# docker-compose.yml - 容器编排文件
# nginx/        - Nginx配置
# deploy.sh     - 部署脚本
```

### 步骤3：通过宝塔终端执行部署

```bash
# 1. 在宝塔面板中打开终端
# 点击左侧菜单 "终端"

# 2. 进入部署目录
cd /opt/blog/deploy-package

# 3. 执行部署脚本
chmod +x deploy.sh
./deploy.sh

# 4. 查看部署状态
docker-compose ps
docker-compose logs
```

## 🚀 方案四：自动化部署脚本

### 步骤1：创建一键部署脚本

创建文件 `one-click-deploy.sh`，复制以下内容：

```bash
#!/bin/bash

echo "=== 博客项目一键部署脚本 ==="

# 配置变量
SERVER_HOST="your-server-ip"
SERVER_USER="your-username"
DEPLOY_PATH="/opt/blog"

# 检查本地构建产物
echo "🔍 检查本地构建产物..."
if [ ! -d "Vue3/dist" ]; then
    echo "❌ 前端构建产物不存在，开始构建前端..."
    cd Vue3
    npm install
    npm run build
    cd ..
fi

if [ ! -f "backend/target/blog-backend-1.0.0.jar" ]; then
    echo "❌ 后端构建产物不存在，开始构建后端..."
    cd backend
    mvn clean package -DskipTests
    cd ..
fi

# 创建部署包
echo "📦 创建部署包..."
./build-deploy-package.sh

# 获取最新的部署包文件名
DEPLOY_FILE=$(ls -t blog-deployment-*.tar.gz | head -1)
echo "📤 准备上传: $DEPLOY_FILE"

# 上传到服务器
echo "🚀 上传部署包到服务器..."
scp $DEPLOY_FILE $SERVER_USER@$SERVER_HOST:$DEPLOY_PATH/

# 远程部署
echo "🔧 执行远程部署..."
ssh $SERVER_USER@$SERVER_HOST << REMOTE_EOF
cd $DEPLOY_PATH
tar -xzf $DEPLOY_FILE
cd deploy-package
chmod +x deploy.sh
./deploy.sh
REMOTE_EOF

echo "✅ 部署完成！"
echo "🌐 访问地址: http://$SERVER_HOST"
echo "🔧 API地址: http://$SERVER_HOST:8080"
```

### 步骤2：配置服务器信息

```bash
# 编辑部署脚本，修改服务器信息
vim one-click-deploy.sh

# 修改以下变量：
# SERVER_HOST="your-server-ip"     # 改为你的服务器IP
# SERVER_USER="your-username"      # 改为你的SSH用户名
# DEPLOY_PATH="/opt/blog"          # 部署路径（可选修改）
```

### 步骤3：执行一键部署

```bash
# 确保SSH密钥已配置
ssh-copy-id your-username@your-server-ip

# 给脚本执行权限
chmod +x one-click-deploy.sh

# 执行一键部署
./one-click-deploy.sh

# 查看部署日志
# 脚本会显示详细的部署过程
# 包括构建、打包、上传、部署各个阶段
```

### 步骤4：验证部署结果

```bash
# 检查服务器状态
ssh your-username@your-server-ip
cd /opt/blog/deploy-package
docker-compose ps

# 测试访问
curl -I http://your-server-ip
curl http://your-server-ip:8080/api/health
```

## 🐳 方案四：Docker 云端构建

### 步骤1：创建云端构建文件

创建文件 `Dockerfile.cloud`，复制以下内容：

```dockerfile
# 多阶段构建 - 前端构建阶段
FROM node:16-alpine AS frontend-builder
WORKDIR /app/frontend
COPY Vue3/package*.json ./
RUN npm ci --only=production
COPY Vue3/ ./
RUN npm run build

# 多阶段构建 - 后端构建阶段
FROM maven:3.8.1-openjdk-8-slim AS backend-builder
WORKDIR /app/backend
COPY backend/pom.xml ./
RUN mvn dependency:go-offline -B
COPY backend/src ./src
RUN mvn clean package -DskipTests

# 生产环境镜像
FROM openjdk:8-jre-alpine
WORKDIR /app

# 安装必要工具
RUN apk add --no-cache curl bash

# 复制构建产物
COPY --from=frontend-builder /app/frontend/dist ./frontend
COPY --from=backend-builder /app/backend/target/*.jar ./backend.jar

# 复制配置文件
COPY backend/src/main/resources/application-prod.yml ./application-prod.yml

# 创建必要目录
RUN mkdir -p uploads/{articles,avatars,comments} logs

# 设置权限
RUN chmod +x backend.jar

# 暴露端口
EXPOSE 8080

# 健康检查
HEALTHCHECK --interval=30s --timeout=3s --start-period=60s --retries=3 \
  CMD curl -f http://localhost:8080/api/health || exit 1

# 启动应用
CMD ["java", "-jar", "-Dspring.profiles.active=prod", "backend.jar"]
```

### 步骤2：创建Docker编排文件

创建文件 `docker-compose.cloud.yml`，复制以下内容：

```yaml
version: '3.8'

services:
  # MySQL 数据库
  mysql:
    image: mysql:8.0.36
    container_name: blog-mysql
    environment:
      MYSQL_ROOT_PASSWORD: ${DB_ROOT_PASSWORD:-123456}
      MYSQL_DATABASE: ${DB_NAME:-musicgame_db}
      MYSQL_USER: ${DB_USER:-blog_user}
      MYSQL_PASSWORD: ${DB_PASSWORD:-blog_password}
    volumes:
      - mysql_data:/var/lib/mysql
      - ./sql:/docker-entrypoint-initdb.d
    ports:
      - "3306:3306"
    networks:
      - blog-network
    restart: unless-stopped
    healthcheck:
      test: ["CMD", "mysqladmin", "ping", "-h", "localhost"]
      timeout: 20s
      retries: 10

  # 博客应用
  blog-app:
    build:
      context: .
      dockerfile: Dockerfile.cloud
    container_name: blog-app
    environment:
      SPRING_PROFILES_ACTIVE: prod
      DB_HOST: mysql
      DB_PORT: 3306
      DB_NAME: ${DB_NAME:-musicgame_db}
      DB_USER: ${DB_USER:-blog_user}
      DB_PASSWORD: ${DB_PASSWORD:-blog_password}
    volumes:
      - uploads_data:/app/uploads
      - logs_data:/app/logs
    ports:
      - "8080:8080"
    networks:
      - blog-network
    depends_on:
      mysql:
        condition: service_healthy
    restart: unless-stopped
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:8080/api/health"]
      interval: 30s
      timeout: 10s
      retries: 3
      start_period: 60s

  # Nginx 反向代理
  nginx:
    image: nginx:alpine
    container_name: blog-nginx
    volumes:
      - ./nginx/nginx.conf:/etc/nginx/nginx.conf:ro
      - ./nginx/conf.d:/etc/nginx/conf.d:ro
    ports:
      - "80:80"
      - "443:443"
    networks:
      - blog-network
    depends_on:
      - blog-app
    restart: unless-stopped

volumes:
  mysql_data:
  uploads_data:
  logs_data:

networks:
  blog-network:
    driver: bridge
```

## 📦 一键部署脚本

### 步骤1：创建部署脚本

创建文件 `cloud-deploy.sh`，复制以下内容：

```bash
#!/bin/bash

set -e

echo "=== 博客项目云端部署脚本 ==="
echo

# 检查必要工具
check_requirements() {
    echo "检查必要工具..."
    
    if ! command -v git &> /dev/null; then
        echo "错误: 未安装 git"
        exit 1
    fi
    
    if ! command -v docker &> /dev/null; then
        echo "错误: 未安装 docker"
        exit 1
    fi
    
    if ! command -v docker-compose &> /dev/null; then
        echo "错误: 未安装 docker-compose"
        exit 1
    fi
    
    echo "✓ 所有必要工具已安装"
}

# 克隆或更新代码
update_code() {
    echo "更新代码..."
    
    if [ -d ".git" ]; then
        echo "更新现有仓库..."
        git pull origin main
    else
        echo "请先克隆项目仓库"
        echo "git clone <your-repo-url> ."
        exit 1
    fi
    
    echo "✓ 代码更新完成"
}

# 构建和部署
deploy() {
    echo "开始构建和部署..."
    
    # 停止现有服务
    echo "停止现有服务..."
    docker-compose -f docker-compose.cloud.yml down
    
    # 清理旧镜像
    echo "清理旧镜像..."
    docker system prune -f
    
    # 构建新镜像
    echo "构建新镜像..."
    docker-compose -f docker-compose.cloud.yml build --no-cache
    
    # 启动服务
    echo "启动服务..."
    docker-compose -f docker-compose.cloud.yml up -d
    
    # 等待服务启动
    echo "等待服务启动..."
    sleep 30
    
    # 检查服务状态
    echo "检查服务状态..."
    docker-compose -f docker-compose.cloud.yml ps
    
    echo "✓ 部署完成"
}

# 健康检查
health_check() {
    echo "执行健康检查..."
    
    # 检查数据库连接
    if docker exec blog-mysql mysqladmin ping -h localhost --silent; then
        echo "✓ 数据库连接正常"
    else
        echo "✗ 数据库连接失败"
        return 1
    fi
    
    # 检查应用健康
    if curl -f http://localhost:8080/api/health > /dev/null 2>&1; then
        echo "✓ 应用服务正常"
    else
        echo "✗ 应用服务异常"
        return 1
    fi
    
    # 检查前端访问
    if curl -f http://localhost > /dev/null 2>&1; then
        echo "✓ 前端访问正常"
    else
        echo "✗ 前端访问异常"
        return 1
    fi
    
    echo "✓ 所有服务健康检查通过"
}

# 显示访问信息
show_info() {
    echo
    echo "=== 部署完成 ==="
    echo "前端访问地址: http://localhost"
    echo "后端API地址: http://localhost:8080/api"
    echo "数据库地址: localhost:3306"
    echo
    echo "管理命令:"
    echo "查看日志: docker-compose -f docker-compose.cloud.yml logs -f"
    echo "停止服务: docker-compose -f docker-compose.cloud.yml down"
    echo "重启服务: docker-compose -f docker-compose.cloud.yml restart"
    echo
}

# 主流程
main() {
    check_requirements
    update_code
    deploy
    
    if health_check; then
        show_info
    else
        echo "部署可能存在问题，请检查日志"
        docker-compose -f docker-compose.cloud.yml logs
        exit 1
    fi
}

# 执行主流程
main "$@"
```

## 🎯 实际操作步骤

### 方案一：GitHub Actions 操作步骤

**第1步：准备代码**
```bash
# 确保代码已提交到 GitHub
git add .
git commit -m "添加 GitHub Actions 部署配置"
git push origin main
```

**第2步：查看构建状态**
1. 进入 GitHub 仓库页面
2. 点击 `Actions` 标签
3. 查看工作流运行状态
4. 点击具体的工作流查看详细日志

**第3步：下载构建产物**
```bash
# 在服务器上下载构建产物
cd /tmp
wget https://github.com/your-username/your-repo/releases/latest/download/blog-deploy-package.tar.gz
tar -xzf blog-deploy-package.tar.gz
chmod +x scripts/deploy.sh
./scripts/deploy.sh
```

### 方案二：Docker 本地构建操作

**第1步：构建镜像**
```bash
# 构建应用镜像
docker build -f Dockerfile.cloud -t blog-app:latest .
```

**第2步：启动服务**
```bash
# 使用 Docker Compose 启动所有服务
docker-compose -f docker-compose.cloud.yml up -d
```

**第3步：检查服务状态**
```bash
# 查看容器状态
docker-compose -f docker-compose.cloud.yml ps

# 查看日志
docker-compose -f docker-compose.cloud.yml logs -f
```

### 方案三：一键部署操作

**第1步：设置脚本权限**
```bash
# 给脚本执行权限
chmod +x cloud-deploy.sh
```

**第2步：执行部署**
```bash
# 运行一键部署脚本
./cloud-deploy.sh
```

**第3步：验证部署**
```bash
# 检查服务状态
curl http://localhost
curl http://localhost:8080/api/health
```

### 方案四：开发环境操作

**GitHub Codespaces：**
1. 进入 GitHub 仓库页面
2. 点击绿色的 `Code` 按钮
3. 选择 `Codespaces` 标签
4. 点击 `Create codespace on main`
5. 等待环境初始化完成

**本地开发容器：**
1. 确保安装了 VS Code 和 Docker
2. 安装 `Dev Containers` 扩展
3. 在 VS Code 中打开项目
4. 按 `Ctrl+Shift+P`，选择 `Dev Containers: Reopen in Container`
5. 等待容器构建完成

## 📊 部署后维护操作

### 创建监控脚本

**第1步：创建监控脚本**
```bash
# 创建监控脚本文件
cat > monitor.sh << 'EOF'
#!/bin/bash
# 云端部署监控脚本

while true; do
    echo "=== $(date) ==="
    
    # 检查容器状态
    echo "容器状态："
    docker-compose -f docker-compose.cloud.yml ps
    
    # 检查资源使用
    echo "资源使用："
    docker stats --no-stream
    
    # 检查磁盘空间
    echo "磁盘空间："
    df -h
    
    # 检查服务健康
    echo "服务健康检查："
    curl -s http://localhost/api/health || echo "API服务异常"
    curl -s http://localhost > /dev/null && echo "前端服务正常" || echo "前端服务异常"
    
    echo "-------------------"
    sleep 300  # 5分钟检查一次
done
EOF

# 设置执行权限
chmod +x monitor.sh
```

**第2步：运行监控**
```bash
# 后台运行监控脚本
nohup ./monitor.sh > monitor.log 2>&1 &

# 查看监控日志
tail -f monitor.log
```

### 常用维护命令

**查看服务状态：**
```bash
# 查看所有容器状态
docker ps -a

# 查看特定服务日志
docker-compose -f docker-compose.cloud.yml logs blog-app
docker-compose -f docker-compose.cloud.yml logs mysql
docker-compose -f docker-compose.cloud.yml logs nginx
```

**重启服务：**
```bash
# 重启所有服务
docker-compose -f docker-compose.cloud.yml restart

# 重启特定服务
docker-compose -f docker-compose.cloud.yml restart blog-app
```

**更新部署：**
```bash
# 拉取最新代码
git pull origin main

# 重新构建和部署
docker-compose -f docker-compose.cloud.yml down
docker-compose -f docker-compose.cloud.yml build --no-cache
docker-compose -f docker-compose.cloud.yml up -d
```

**数据备份：**
```bash
# 备份数据库
docker exec blog-mysql mysqldump -u root -p123456 musicgame_db > backup_$(date +%Y%m%d).sql

# 备份上传文件
tar -czf uploads_backup_$(date +%Y%m%d).tar.gz uploads/
```

## ✅ 部署验证清单

完成部署后，请按以下清单验证：

- [ ] **容器状态检查**
  ```bash
  docker-compose -f docker-compose.cloud.yml ps
  ```

- [ ] **前端访问测试**
  ```bash
  curl http://localhost
  ```

- [ ] **后端API测试**
  ```bash
  curl http://localhost:8080/api/health
  ```

- [ ] **数据库连接测试**
  ```bash
  docker exec blog-mysql mysql -u root -p123456 -e "SHOW DATABASES;"
  ```

- [ ] **文件上传测试**
  - 登录系统
  - 尝试上传头像或文章图片
  - 检查文件是否正常显示

- [ ] **功能完整性测试**
  - [ ] 用户注册登录
  - [ ] 文章发布和查看
  - [ ] 评论功能
  - [ ] 音乐播放
  - [ ] 游戏视频播放

## 🚨 故障排除

**常见问题及解决方案：**

1. **容器启动失败**
   ```bash
   # 查看详细错误日志
   docker-compose -f docker-compose.cloud.yml logs
   
   # 检查端口占用
   netstat -tlnp | grep :8080
   ```

2. **数据库连接失败**
   ```bash
   # 检查MySQL容器状态
   docker exec blog-mysql mysql -u root -p123456 -e "SELECT 1;"
   
   # 重启MySQL容器
   docker-compose -f docker-compose.cloud.yml restart mysql
   ```

3. **前端页面无法访问**
   ```bash
   # 检查Nginx配置
   docker exec blog-nginx nginx -t
   
   # 重载Nginx配置
   docker exec blog-nginx nginx -s reload
   ```

选择适合您需求的方案，按照操作步骤进行部署。建议先在测试环境验证，确认无误后再部署到生产环境。