# 服务器路径排查指南

## 🔍 常见路径问题类型

### 1. 文件上传路径问题
**问题现象：**
- 文件上传失败
- 上传后无法访问
- 404错误

**排查命令：**
```bash
# 检查上传目录是否存在
ls -la /www/wwwroot/blog/uploads/

# 检查目录权限
ls -ld /www/wwwroot/blog/uploads/

# 检查子目录
ls -la /www/wwwroot/blog/uploads/articles/
ls -la /www/wwwroot/blog/uploads/avatars/
ls -la /www/wwwroot/blog/uploads/comments/
```

**解决方案：**
```bash
# 创建缺失目录
mkdir -p /www/wwwroot/blog/uploads/{articles,avatars,comments}

# 设置正确权限
chown -R www:www /www/wwwroot/blog/uploads/
chmod -R 755 /www/wwwroot/blog/uploads/
```

### 2. 静态资源路径问题
**问题现象：**
- 图片、CSS、JS文件加载失败
- 控制台出现404错误

**排查命令：**
```bash
# 检查前端构建产物
ls -la /www/wwwroot/blog/frontend/

# 检查静态资源目录
ls -la /www/wwwroot/blog/frontend/static/
ls -la /www/wwwroot/blog/frontend/images/
ls -la /www/wwwroot/blog/frontend/games/
ls -la /www/wwwroot/blog/frontend/music/
ls -la /www/wwwroot/blog/frontend/videos/
```

**Nginx配置检查：**
```bash
# 检查Nginx配置
cat /www/server/nginx/conf/vhost/likeusewin10.xyz.conf

# 测试Nginx配置
nginx -t

# 重载Nginx配置
nginx -s reload
```

### 3. 后端JAR包路径问题
**问题现象：**
- 后端服务启动失败
- 找不到JAR文件

**排查命令：**
```bash
# 检查JAR文件是否存在
ls -la /www/wwwroot/blog/backend/

# 检查JAR文件权限
ls -l /www/wwwroot/blog/backend/blog-backend-1.0.0.jar

# 检查Java环境
java -version
which java
```

### 4. 配置文件路径问题
**问题现象：**
- 应用无法读取配置
- 数据库连接失败

**排查命令：**
```bash
# 检查配置文件
ls -la /www/wwwroot/blog/backend/application-prod.yml

# 查看配置内容
cat /www/wwwroot/blog/backend/application-prod.yml

# 检查日志目录
ls -la /www/wwwroot/blog/logs/
```

## 🛠️ 快速排查脚本

### 创建路径检查脚本
```bash
# 创建检查脚本
cat > /www/wwwroot/blog/check-paths.sh << 'EOF'
#!/bin/bash

echo "=== 博客项目路径检查 ==="
echo

# 检查主目录
echo "1. 检查主目录结构："
ls -la /www/wwwroot/blog/
echo

# 检查前端目录
echo "2. 检查前端目录："
if [ -d "/www/wwwroot/blog/frontend" ]; then
    echo "✓ 前端目录存在"
    ls -la /www/wwwroot/blog/frontend/ | head -10
else
    echo "✗ 前端目录不存在"
fi
echo

# 检查后端目录
echo "3. 检查后端目录："
if [ -d "/www/wwwroot/blog/backend" ]; then
    echo "✓ 后端目录存在"
    ls -la /www/wwwroot/blog/backend/
else
    echo "✗ 后端目录不存在"
fi
echo

# 检查上传目录
echo "4. 检查上传目录："
if [ -d "/www/wwwroot/blog/uploads" ]; then
    echo "✓ 上传目录存在"
    ls -la /www/wwwroot/blog/uploads/
else
    echo "✗ 上传目录不存在"
fi
echo

# 检查日志目录
echo "5. 检查日志目录："
if [ -d "/www/wwwroot/blog/logs" ]; then
    echo "✓ 日志目录存在"
    ls -la /www/wwwroot/blog/logs/
else
    echo "✗ 日志目录不存在"
fi
echo

# 检查关键文件
echo "6. 检查关键文件："
files=(
    "/www/wwwroot/blog/backend/blog-backend-1.0.0.jar"
    "/www/wwwroot/blog/backend/application-prod.yml"
    "/www/wwwroot/blog/frontend/index.html"
    "/www/server/nginx/conf/vhost/likeusewin10.xyz.conf"
)

for file in "${files[@]}"; do
    if [ -f "$file" ]; then
        echo "✓ $file 存在"
    else
        echo "✗ $file 不存在"
    fi
done
echo

# 检查权限
echo "7. 检查关键目录权限："
ls -ld /www/wwwroot/blog/
ls -ld /www/wwwroot/blog/uploads/ 2>/dev/null || echo "uploads目录不存在"
ls -ld /www/wwwroot/blog/logs/ 2>/dev/null || echo "logs目录不存在"
echo

# 检查进程
echo "8. 检查相关进程："
ps aux | grep -E "(java|nginx|mysql)" | grep -v grep
echo

echo "=== 检查完成 ==="
EOF

# 设置执行权限
chmod +x /www/wwwroot/blog/check-paths.sh
```

### 使用检查脚本
```bash
# 运行路径检查
/www/wwwroot/blog/check-paths.sh
```

## 📋 路径配置标准化

### 标准目录结构
```
/www/wwwroot/blog/
├── frontend/              # 前端构建产物
│   ├── index.html
│   ├── static/           # 静态资源
│   ├── images/           # 图片资源
│   ├── games/            # 游戏资源
│   ├── music/            # 音乐资源
│   └── videos/           # 视频资源
├── backend/              # 后端应用
│   ├── blog-backend-1.0.0.jar
│   └── application-prod.yml
├── uploads/              # 上传文件
│   ├── articles/         # 文章图片
│   ├── avatars/          # 用户头像
│   └── comments/         # 评论图片
├── logs/                 # 应用日志
│   ├── application.log
│   └── error.log
├── mysql/                # MySQL数据
│   └── data/
├── scripts/              # 管理脚本
│   ├── start.sh
│   ├── stop.sh
│   └── check-paths.sh
└── backup/               # 备份文件
    ├── database/
    └── uploads/
```

### 环境变量配置
```bash
# 在 ~/.bashrc 或 /etc/profile 中添加
export BLOG_HOME="/www/wwwroot/blog"
export BLOG_FRONTEND="$BLOG_HOME/frontend"
export BLOG_BACKEND="$BLOG_HOME/backend"
export BLOG_UPLOADS="$BLOG_HOME/uploads"
export BLOG_LOGS="$BLOG_HOME/logs"

# 重新加载环境变量
source ~/.bashrc
```

## 🔧 自动修复脚本

### 创建自动修复脚本
```bash
cat > /www/wwwroot/blog/fix-paths.sh << 'EOF'
#!/bin/bash

echo "=== 自动修复路径问题 ==="

# 创建必要目录
echo "创建必要目录..."
mkdir -p /www/wwwroot/blog/{frontend,backend,uploads/{articles,avatars,comments},logs,mysql/data,scripts,backup/{database,uploads}}

# 设置目录权限
echo "设置目录权限..."
chown -R www:www /www/wwwroot/blog/uploads/
chown -R www:www /www/wwwroot/blog/logs/
chmod -R 755 /www/wwwroot/blog/uploads/
chmod -R 755 /www/wwwroot/blog/logs/

# 检查并修复Nginx配置中的路径
echo "检查Nginx配置..."
NGINX_CONF="/www/server/nginx/conf/vhost/likeusewin10.xyz.conf"
if [ -f "$NGINX_CONF" ]; then
    # 备份原配置
    cp "$NGINX_CONF" "$NGINX_CONF.backup.$(date +%Y%m%d_%H%M%S)"
    echo "Nginx配置已备份"
fi

# 检查应用配置文件
echo "检查应用配置..."
CONFIG_FILE="/www/wwwroot/blog/backend/application-prod.yml"
if [ -f "$CONFIG_FILE" ]; then
    echo "应用配置文件存在"
else
    echo "警告：应用配置文件不存在，请检查后端部署"
fi

echo "=== 修复完成 ==="
echo "请运行 check-paths.sh 验证修复结果"
EOF

chmod +x /www/wwwroot/blog/fix-paths.sh
```

## 📊 路径问题诊断清单

### 快速诊断命令
```bash
# 一键诊断
echo "=== 快速路径诊断 ==="
echo "1. 检查磁盘空间："
df -h /www/wwwroot/blog/

echo "2. 检查inode使用："
df -i /www/wwwroot/blog/

echo "3. 检查目录权限："
namei -l /www/wwwroot/blog/uploads/articles/

echo "4. 检查SELinux状态："
getenforce 2>/dev/null || echo "SELinux未安装"

echo "5. 检查文件系统错误："
dmesg | tail -10 | grep -i error
```

### 常用排查命令速查
```bash
# 查找大文件
find /www/wwwroot/blog/ -type f -size +100M -exec ls -lh {} \;

# 查找权限异常文件
find /www/wwwroot/blog/ -type f ! -perm 644 -exec ls -l {} \;

# 查找空目录
find /www/wwwroot/blog/ -type d -empty

# 检查符号链接
find /www/wwwroot/blog/ -type l -exec ls -l {} \;

# 查看最近修改的文件
find /www/wwwroot/blog/ -type f -mtime -1 -exec ls -lt {} \;
```

## 🚨 紧急恢复方案

### 路径完全重建
```bash
#!/bin/bash
# 紧急重建脚本

echo "警告：此操作将重建整个目录结构"
read -p "确认继续？(y/N): " confirm
if [[ $confirm != [yY] ]]; then
    exit 1
fi

# 备份现有数据
echo "备份现有数据..."
tar -czf "/tmp/blog-backup-$(date +%Y%m%d_%H%M%S).tar.gz" /www/wwwroot/blog/ 2>/dev/null

# 重建目录结构
echo "重建目录结构..."
rm -rf /www/wwwroot/blog/
mkdir -p /www/wwwroot/blog/{frontend,backend,uploads/{articles,avatars,comments},logs,mysql/data,scripts,backup/{database,uploads}}

# 设置权限
chown -R www:www /www/wwwroot/blog/
chmod -R 755 /www/wwwroot/blog/

echo "目录结构重建完成"
echo "请重新部署应用文件"
```

使用此指南可以快速定位和解决服务器部署中的各种路径问题。建议将检查脚本加入到日常维护流程中。