# 个人博客项目阿里云宝塔面板部署指南

## 项目概述

本项目是一个基于 Vue3 + Spring Boot + MySQL 的个人博客系统，现在需要部署到阿里云宝塔面板服务器上。

### 技术栈
- **前端**: Vue3 + Vue Router
- **后端**: Spring Boot 2.7 + MyBatis Plus
- **数据库**: MySQL 8.0 (Docker容器部署)
- **部署**: 宝塔面板 + Docker

## 部署环境差异说明

### 本地环境 vs 服务器环境
- **本地**: MySQL直接安装
- **服务器**: MySQL部署在Docker容器中
- **部署方式**: 使用宝塔面板进行管理

## 第一步：本地项目编译打包

### 1.1 前端Vue3项目编译

```bash
# 进入前端项目目录
cd Vue3

# 安装依赖（如果还没有安装）
npm install

# 清理之前的构建文件
rm -rf dist

# 构建生产版本
npm run build

# 验证构建结果
ls -la dist/
```

构建完成后，`dist` 目录包含：
- `index.html` - 主页面
- `static/` - 静态资源（CSS、JS、图片等）
- `games/` - 游戏相关资源
- `images/` - 图片资源
- `music/` - 音乐资源
- `videos/` - 视频资源

### 1.2 后端Spring Boot项目编译

```bash
# 进入后端项目目录
cd backend

# 清理之前的构建文件
mvn clean

# 编译并打包（跳过测试）
mvn package -DskipTests

# 验证构建结果
ls -la target/
```

构建完成后，`target` 目录包含：
- `blog-backend-1.0.0.jar` - 可执行JAR文件
- `classes/` - 编译后的类文件

## 第二步：服务器环境准备

### 2.1 宝塔面板基础环境

确保宝塔面板已安装以下组件：
- **Docker** - 用于MySQL容器部署
- **Java** - JDK 1.8+
- **Nginx** - Web服务器
- **文件管理器** - 用于上传和管理文件

### 2.2 创建项目目录结构

在宝塔面板文件管理器中创建以下目录结构：

```
/www/wwwroot/personal-blog/
├── frontend/          # 前端文件
├── backend/           # 后端文件
├── uploads/           # 上传文件目录
│   ├── articles/      # 文章图片
│   ├── avatars/       # 用户头像
│   └── comments/      # 评论图片
├── logs/              # 日志目录
├── sql/               # 数据库脚本
└── config/            # 配置文件
```

### 2.3 MySQL Docker容器部署

在宝塔面板终端中执行：

```bash
# 创建MySQL数据卷
docker volume create personal-blog-mysql-data

# 启动MySQL容器
docker run -d \
  --name personal-blog-mysql \
  --restart always \
  -e MYSQL_ROOT_PASSWORD=root123456 \
  -e MYSQL_DATABASE=musicgame_db \
  -e MYSQL_USER=musicgame_user \
  -e MYSQL_PASSWORD=musicgame123 \
  -p 3306:3306 \
  -v personal-blog-mysql-data:/var/lib/mysql \
  mysql:8.0.36 \
  --default-authentication-plugin=mysql_native_password

# 验证容器启动
docker ps | grep personal-blog-mysql
```

## 第三步：文件上传

### 3.1 上传前端文件

使用宝塔面板文件管理器或FTP工具：

1. 将本地 `Vue3/dist/` 目录下的所有文件上传到服务器 `/www/wwwroot/personal-blog/frontend/`
2. 确保文件权限设置为 755

### 3.2 上传后端文件

1. 将 `backend/target/blog-backend-1.0.0.jar` 上传到 `/www/wwwroot/personal-blog/backend/`
2. 将 `backend/src/main/resources/application-prod.yml` 上传到同一目录

### 3.3 上传数据库脚本

将 `sql/` 目录下的所有SQL文件上传到 `/www/wwwroot/personal-blog/sql/`

### 3.4 创建必要目录

```bash
# 在服务器上创建上传目录
mkdir -p /www/wwwroot/personal-blog/uploads/{articles,avatars,comments}
mkdir -p /www/wwwroot/personal-blog/logs

# 设置权限
chmod -R 755 /www/wwwroot/personal-blog/uploads
chmod -R 755 /www/wwwroot/personal-blog/logs
```

## 第四步：数据库初始化

### 4.1 导入数据库结构

```bash
# 进入MySQL容器
docker exec -it personal-blog-mysql mysql -u root -proot123456

# 选择数据库
USE musicgame_db;

# 退出容器
exit

# 从外部导入SQL文件
docker exec -i personal-blog-mysql mysql -u musicgame_user -pmusicgame123 musicgame_db < /www/wwwroot/personal-blog/sql/创建结构.sql

# 导入测试数据（可选）
docker exec -i personal-blog-mysql mysql -u musicgame_user -pmusicgame123 musicgame_db < /www/wwwroot/personal-blog/sql/插入测试数据.sql
```

### 4.2 验证数据库

```bash
# 检查数据库和表
docker exec -it personal-blog-mysql mysql -u musicgame_user -pmusicgame123 musicgame_db -e "SHOW TABLES;"
```

## 第五步：后端应用配置

### 5.1 创建生产环境配置文件

在 `/www/wwwroot/personal-blog/backend/` 目录创建 `application-prod.yml`：

```yaml
server:
  port: 8080
  servlet:
    context-path: /api

spring:
  datasource:
    url: jdbc:mysql://localhost:3306/musicgame_db?useUnicode=true&characterEncoding=utf8&zeroDateTimeBehavior=convertToNull&useSSL=false&serverTimezone=GMT%2B8
    username: musicgame_user
    password: musicgame123
    driver-class-name: com.mysql.cj.jdbc.Driver
  
  servlet:
    multipart:
      max-file-size: 10MB
      max-request-size: 10MB

# 文件上传路径配置
file:
  upload:
    path: /www/wwwroot/personal-blog/uploads/

# 日志配置
logging:
  level:
    com.likeusewin10: DEBUG
  file:
    name: /www/wwwroot/personal-blog/logs/application.log
  pattern:
    file: "%d{yyyy-MM-dd HH:mm:ss} [%thread] %-5level %logger{36} - %msg%n"

# MyBatis Plus配置
mybatis-plus:
  configuration:
    log-impl: org.apache.ibatis.logging.stdout.StdOutImpl
  global-config:
    db-config:
      logic-delete-field: deleted
      logic-delete-value: 1
      logic-not-delete-value: 0
```

### 5.2 创建启动脚本

在 `/www/wwwroot/personal-blog/backend/` 目录创建 `start.sh`：

```bash
#!/bin/bash

# 设置Java环境变量
export JAVA_HOME=/usr/lib/jvm/java-8-openjdk-amd64
export PATH=$JAVA_HOME/bin:$PATH

# 应用配置
APP_NAME="personal-blog-backend"
JAR_FILE="blog-backend-1.0.0.jar"
PID_FILE="/tmp/${APP_NAME}.pid"
LOG_FILE="/www/wwwroot/personal-blog/logs/startup.log"

# JVM参数
JVM_OPTS="-Xms512m -Xmx1024m -XX:+UseG1GC"

# Spring Boot参数
SPRING_OPTS="--spring.profiles.active=prod"

echo "Starting $APP_NAME..."

# 检查是否已经运行
if [ -f $PID_FILE ]; then
    PID=$(cat $PID_FILE)
    if ps -p $PID > /dev/null 2>&1; then
        echo "$APP_NAME is already running (PID: $PID)"
        exit 1
    else
        rm -f $PID_FILE
    fi
fi

# 启动应用
nohup java $JVM_OPTS -jar $JAR_FILE $SPRING_OPTS > $LOG_FILE 2>&1 &
echo $! > $PID_FILE

echo "$APP_NAME started successfully!"
echo "PID: $(cat $PID_FILE)"
echo "Log file: $LOG_FILE"
```

创建停止脚本 `stop.sh`：

```bash
#!/bin/bash

APP_NAME="personal-blog-backend"
PID_FILE="/tmp/${APP_NAME}.pid"

if [ -f $PID_FILE ]; then
    PID=$(cat $PID_FILE)
    if ps -p $PID > /dev/null 2>&1; then
        echo "Stopping $APP_NAME (PID: $PID)..."
        kill $PID
        
        # 等待进程结束
        for i in {1..30}; do
            if ! ps -p $PID > /dev/null 2>&1; then
                echo "$APP_NAME stopped successfully!"
                rm -f $PID_FILE
                exit 0
            fi
            sleep 1
        done
        
        # 强制结束
        echo "Force killing $APP_NAME..."
        kill -9 $PID
        rm -f $PID_FILE
        echo "$APP_NAME force stopped!"
    else
        echo "$APP_NAME is not running"
        rm -f $PID_FILE
    fi
else
    echo "$APP_NAME is not running (no PID file found)"
fi
```

设置脚本权限：

```bash
chmod +x /www/wwwroot/personal-blog/backend/start.sh
chmod +x /www/wwwroot/personal-blog/backend/stop.sh
```

## 第六步：Nginx配置

### 6.1 在宝塔面板中创建网站

1. 登录宝塔面板
2. 点击「网站」→「添加站点」
3. 填写域名：`likeusewin10.xyz`
4. 根目录：`/www/wwwroot/personal-blog/frontend`
5. 创建站点

### 6.2 配置Nginx反向代理

在宝塔面板中编辑网站的Nginx配置：

```nginx
server {
    listen 80;
    server_name likeusewin10.xyz;
    root /www/wwwroot/personal-blog/frontend;
    index index.html;
    
    # 客户端最大上传文件大小
    client_max_body_size 10M;
    
    # 前端静态文件
    location / {
        try_files $uri $uri/ /index.html;
        
        # 静态文件缓存
        location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot)$ {
            expires 1y;
            add_header Cache-Control "public, immutable";
        }
    }
    
    # API代理到后端
    location /api/ {
        proxy_pass http://127.0.0.1:8080/api/;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        
        # 超时设置
        proxy_connect_timeout 60s;
        proxy_send_timeout 60s;
        proxy_read_timeout 60s;
    }
    
    # 上传文件访问
    location /uploads/ {
        alias /www/wwwroot/personal-blog/uploads/;
        expires 1d;
        add_header Cache-Control "public";
    }
    
    # Gzip压缩
    gzip on;
    gzip_vary on;
    gzip_min_length 1024;
    gzip_types text/plain text/css text/xml text/javascript application/javascript application/xml+rss application/json;
    
    # 安全头
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header X-Content-Type-Options "nosniff" always;
}
```

## 第七步：启动服务

### 7.1 启动后端服务

```bash
# 进入后端目录
cd /www/wwwroot/personal-blog/backend

# 启动服务
./start.sh

# 检查启动日志
tail -f /www/wwwroot/personal-blog/logs/startup.log
```

### 7.2 重启Nginx

在宝塔面板中重启Nginx服务，或使用命令：

```bash
nginx -t  # 检查配置
nginx -s reload  # 重新加载配置
```

## 第八步：验证部署

### 8.1 检查服务状态

```bash
# 检查MySQL容器
docker ps | grep personal-blog-mysql

# 检查后端进程
ps aux | grep java

# 检查端口监听
netstat -tlnp | grep :8080
netstat -tlnp | grep :3306
```

### 8.2 功能测试

1. **前端访问测试**：
   - 访问 `http://likeusewin10.xyz`
   - 检查页面是否正常加载
   - 测试各个页面跳转

2. **后端API测试**：
   ```bash
   # 健康检查
   curl http://likeusewin10.xyz/api/health
   
   # 获取文章列表
   curl http://likeusewin10.xyz/api/articles
   ```

3. **数据库连接测试**：
   ```bash
   # 检查数据库连接
   docker exec -it personal-blog-mysql mysql -u musicgame_user -pmusicgame123 -e "SELECT 1;"
   ```

## 第九步：设置开机自启

### 9.1 创建系统服务

创建 `/etc/systemd/system/personal-blog.service`：

```ini
[Unit]
Description=Personal Blog Backend Service
After=network.target docker.service
Requires=docker.service

[Service]
Type=forking
User=root
WorkingDirectory=/www/wwwroot/personal-blog/backend
ExecStart=/www/wwwroot/personal-blog/backend/start.sh
ExecStop=/www/wwwroot/personal-blog/backend/stop.sh
Restart=always
RestartSec=10

[Install]
WantedBy=multi-user.target
```

启用服务：

```bash
# 重新加载systemd配置
systemctl daemon-reload

# 启用服务
systemctl enable personal-blog

# 启动服务
systemctl start personal-blog

# 检查状态
systemctl status personal-blog
```

### 9.2 MySQL容器自启动

MySQL容器已经设置了 `--restart always`，会自动重启。

## 第十步：监控和维护

### 10.1 日志监控

```bash
# 查看应用日志
tail -f /www/wwwroot/personal-blog/logs/application.log

# 查看启动日志
tail -f /www/wwwroot/personal-blog/logs/startup.log

# 查看MySQL日志
docker logs personal-blog-mysql
```

### 10.2 性能监控

```bash
# 系统资源监控
top
df -h
free -h

# Docker容器监控
docker stats personal-blog-mysql
```

### 10.3 备份策略

创建备份脚本 `/www/wwwroot/personal-blog/backup.sh`：

```bash
#!/bin/bash

BACKUP_DIR="/www/backup/personal-blog"
DATE=$(date +%Y%m%d_%H%M%S)

# 创建备份目录
mkdir -p $BACKUP_DIR

# 备份数据库
docker exec personal-blog-mysql mysqldump -u root -proot123456 musicgame_db > $BACKUP_DIR/database_$DATE.sql

# 备份上传文件
tar -czf $BACKUP_DIR/uploads_$DATE.tar.gz /www/wwwroot/personal-blog/uploads/

# 清理7天前的备份
find $BACKUP_DIR -name "*.sql" -mtime +7 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +7 -delete

echo "Backup completed: $DATE"
```

设置定时备份：

```bash
# 编辑crontab
crontab -e

# 添加每天凌晨2点备份
0 2 * * * /www/wwwroot/personal-blog/backup.sh
```

## 常见问题解决

### 问题1：后端启动失败

**症状**：Java应用无法启动

**解决方案**：
1. 检查Java版本：`java -version`
2. 检查端口占用：`netstat -tlnp | grep :8080`
3. 查看启动日志：`tail -f /www/wwwroot/personal-blog/logs/startup.log`
4. 检查配置文件路径和权限

### 问题2：数据库连接失败

**症状**：应用无法连接MySQL

**解决方案**：
1. 检查MySQL容器状态：`docker ps | grep mysql`
2. 检查数据库连接：`docker exec -it personal-blog-mysql mysql -u musicgame_user -pmusicgame123`
3. 检查防火墙设置
4. 验证数据库配置参数

### 问题3：文件上传失败

**症状**：无法上传图片或文件

**解决方案**：
1. 检查上传目录权限：`ls -la /www/wwwroot/personal-blog/uploads/`
2. 设置正确权限：`chmod -R 755 /www/wwwroot/personal-blog/uploads/`
3. 检查Nginx文件大小限制
4. 查看应用日志中的错误信息

### 问题4：前端页面无法访问

**症状**：网站无法打开或显示异常

**解决方案**：
1. 检查Nginx配置：`nginx -t`
2. 检查网站根目录文件：`ls -la /www/wwwroot/personal-blog/frontend/`
3. 查看Nginx错误日志
4. 检查域名解析

## 部署完成检查清单

- [ ] MySQL容器正常运行
- [ ] 数据库结构和数据导入完成
- [ ] 后端Java应用正常启动
- [ ] Nginx配置正确并重新加载
- [ ] 前端文件上传完成
- [ ] 网站可以正常访问
- [ ] API接口响应正常
- [ ] 文件上传功能正常
- [ ] 各个页面功能正常
- [ ] 开机自启动配置完成
- [ ] 备份策略配置完成

## 访问地址

部署完成后的访问地址：
- **网站首页**: http://likeusewin10.xyz
- **API接口**: http://likeusewin10.xyz/api
- **上传文件**: http://likeusewin10.xyz/uploads

---

**恭喜！您的个人博客项目已成功部署到阿里云宝塔面板！** 🎉

如有问题，请检查日志文件或联系技术支持。